<?php

namespace App\Console\Commands;

use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;

class CreateAdminUser extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'admin:create 
                            {--name= : Admin name}
                            {--email= : Admin email}
                            {--password= : Admin password}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a new admin user';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $name = $this->option('name') ?? $this->ask('Enter admin name', 'System Administrator');
        $email = $this->option('email') ?? $this->ask('Enter admin email', 'admin@epoe.com');
        $password = $this->option('password') ?? $this->secret('Enter admin password (min 8 characters)');

        // Validate inputs
        $validator = Validator::make([
            'name' => $name,
            'email' => $email,
            'password' => $password,
        ], [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email',
            'password' => 'required|string|min:8',
        ]);

        if ($validator->fails()) {
            $this->error('Validation failed:');
            foreach ($validator->errors()->all() as $error) {
                $this->error('  - ' . $error);
            }
            return 1;
        }

        // Check if user already exists
        $existingUser = User::where('email', $email)->first();
        if ($existingUser) {
            if ($this->confirm("User with email {$email} already exists. Do you want to update it to admin?", false)) {
                $existingUser->update([
                    'name' => $name,
                    'password' => Hash::make($password),
                    'role' => 'admin',
                    'status' => 'active',
                    'email_verified_at' => now(),
                ]);
                $this->info("Admin user updated successfully!");
                $this->table(
                    ['Field', 'Value'],
                    [
                        ['Name', $name],
                        ['Email', $email],
                        ['Password', '******'],
                        ['Role', 'admin'],
                    ]
                );
                return 0;
            } else {
                $this->warn('Operation cancelled.');
                return 1;
            }
        }

        // Create admin user
        User::create([
            'name' => $name,
            'email' => $email,
            'password' => Hash::make($password),
            'role' => 'admin',
            'status' => 'active',
            'email_verified_at' => now(),
        ]);

        $this->info('Admin user created successfully!');
        $this->table(
            ['Field', 'Value'],
            [
                ['Name', $name],
                ['Email', $email],
                ['Password', '******'],
                ['Role', 'admin'],
            ]
        );

        return 0;
    }
}
