<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\User;
use App\Models\Term;
use App\Models\Department;
use App\Models\Level;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Models\Setting;
use App\Services\ActiveTermService;
use App\Services\QueryOptimizer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\View;

class ReportController extends Controller
{
    /**
     * Display reports dashboard
     */
    public function index()
    {
        $activeTerm = ActiveTermService::getActiveTerm();
        $terms = Term::orderBy('start_date', 'desc')->get();
        $departments = Department::orderBy('name')->get();
        $levels = Level::orderBy('name')->get();

        // Quick stats for reports
        $quickStats = [
            'total_submissions' => PoeSubmission::count(),
            'total_students' => User::where('role', 'student')->count(),
            'total_classes' => SchoolClass::count(),
            'total_units' => Unit::count(),
        ];

        return view('admin.reports.index', compact('activeTerm', 'terms', 'departments', 'levels', 'quickStats'));
    }

    /**
     * Generate submission report
     */
    public function submissions(Request $request)
    {
        $request->validate([
            'term_id' => ['nullable', 'exists:terms,id'],
            'department_id' => ['nullable', 'exists:departments,id'],
            'level_id' => ['nullable', 'exists:levels,id'],
            'class_id' => ['nullable', 'exists:classes,id'],
            'status' => ['nullable', 'in:draft,submitted,under_review,trainer_reviewed,hod_approved,exams_verified,validator_approved,rejected'],
            'format' => ['required', 'in:html,pdf,excel'],
        ]);

        // Use optimized query
        $filters = [
            'term_id' => $request->term_id,
            'department_id' => $request->department_id,
            'level_id' => $request->level_id,
            'class_id' => $request->class_id,
            'status' => $request->status,
        ];

        $query = QueryOptimizer::submissionsQuery($filters);
        
        // For reports, load full relationships
        $submissions = $query->with([
            'student:id,name,email',
            'unit:id,name,code',
            'schoolClass:id,name,term_id,department_id,level_id' => [
                'term:id,name',
                'department:id,name',
                'level:id,name'
            ],
            'evidence:id,poe_submission_id,file_name,file_type,file_size',
            'reviews:id,poe_submission_id,reviewer_id,marks,feedback,status' => [
                'reviewer:id,name'
            ],
            'validation:id,poe_submission_id,validator_id,status,comments' => [
                'validator:id,name'
            ]
        ])
        ->withCount('evidence')
        ->latest('poe_submissions.created_at')
        ->get();
        
        $term = $request->term_id ? Term::find($request->term_id) : ActiveTermService::getActiveTerm();
        $department = $request->department_id ? Department::find($request->department_id) : null;
        $level = $request->level_id ? Level::find($request->level_id) : null;
        $class = $request->class_id ? SchoolClass::find($request->class_id) : null;

        $reportData = [
            'submissions' => $submissions,
            'term' => $term,
            'department' => $department,
            'level' => $level,
            'class' => $class,
            'filters' => $request->only(['term_id', 'department_id', 'level_id', 'class_id', 'status']),
        ];

        // Get institution settings for header/footer
        $institution = $this->getInstitutionSettings();

        if ($request->format === 'pdf') {
            return $this->generatePdf('admin.reports.submissions', $reportData, $institution);
        } elseif ($request->format === 'excel') {
            return $this->generateExcel($submissions, 'submissions', $institution);
        }

        return view('admin.reports.submissions', array_merge($reportData, ['institution' => $institution]));
    }

    /**
     * Generate student report
     */
    public function student(Request $request, User $student)
    {
        $request->validate([
            'term_id' => ['nullable', 'exists:terms,id'],
            'format' => ['required', 'in:html,pdf,excel'],
        ]);

        $query = PoeSubmission::where('student_id', $student->id)
            ->with(['unit', 'schoolClass.term', 'schoolClass.department', 'schoolClass.level', 'evidence', 'reviews', 'validation']);

        if ($request->term_id) {
            $query->whereHas('schoolClass', function($q) use ($request) {
                $q->where('term_id', $request->term_id);
            });
        } else {
            $activeTermId = ActiveTermService::getActiveTermId();
            if ($activeTermId) {
                $query->whereHas('schoolClass', function($q) use ($activeTermId) {
                    $q->where('term_id', $activeTermId);
                });
            }
        }

        $submissions = $query->latest()->get();
        $term = $request->term_id ? Term::find($request->term_id) : ActiveTermService::getActiveTerm();

        $reportData = [
            'student' => $student,
            'submissions' => $submissions,
            'term' => $term,
        ];

        $institution = $this->getInstitutionSettings();

        if ($request->format === 'pdf') {
            return $this->generatePdf('admin.reports.student', $reportData, $institution);
        } elseif ($request->format === 'excel') {
            return $this->generateExcel($submissions, 'student_' . $student->id, $institution);
        }

        return view('admin.reports.student', array_merge($reportData, ['institution' => $institution]));
    }

    /**
     * Generate class report
     */
    public function class(Request $request, SchoolClass $class)
    {
        $request->validate([
            'format' => ['required', 'in:html,pdf,excel'],
        ]);

        $submissions = PoeSubmission::where('class_id', $class->id)
            ->with(['student', 'unit', 'evidence', 'reviews', 'validation'])
            ->withCount('evidence')
            ->latest()
            ->get();

        $reportData = [
            'class' => $class->load(['term', 'department', 'level']),
            'submissions' => $submissions,
            'enrollments' => $class->enrollments()->with('student')->get(),
        ];

        $institution = $this->getInstitutionSettings();

        if ($request->format === 'pdf') {
            return $this->generatePdf('admin.reports.class', $reportData, $institution);
        } elseif ($request->format === 'excel') {
            return $this->generateExcel($submissions, 'class_' . $class->id, $institution);
        }

        return view('admin.reports.class', array_merge($reportData, ['institution' => $institution]));
    }

    /**
     * Generate department report
     */
    public function department(Request $request, Department $department)
    {
        $request->validate([
            'term_id' => ['nullable', 'exists:terms,id'],
            'format' => ['required', 'in:html,pdf,excel'],
        ]);

        $term = $request->term_id ? Term::find($request->term_id) : ActiveTermService::getActiveTerm();
        
        $query = PoeSubmission::whereHas('schoolClass', function($q) use ($department, $term) {
            $q->where('department_id', $department->id);
            if ($term) {
                $q->where('term_id', $term->id);
            }
        })->with(['student', 'unit', 'schoolClass', 'evidence', 'reviews', 'validation']);

        $submissions = $query->withCount('evidence')->latest()->get();
        $classes = SchoolClass::where('department_id', $department->id)
            ->when($term, fn($q) => $q->where('term_id', $term->id))
            ->with(['term', 'level'])
            ->get();

        $reportData = [
            'department' => $department,
            'term' => $term,
            'submissions' => $submissions,
            'classes' => $classes,
            'stats' => [
                'total_students' => User::where('department_id', $department->id)->where('role', 'student')->count(),
                'total_classes' => $classes->count(),
                'total_submissions' => $submissions->count(),
                'approved_submissions' => $submissions->where('status', 'validator_approved')->count(),
            ],
        ];

        $institution = $this->getInstitutionSettings();

        if ($request->format === 'pdf') {
            return $this->generatePdf('admin.reports.department', $reportData, $institution);
        } elseif ($request->format === 'excel') {
            return $this->generateExcel($submissions, 'department_' . $department->id, $institution);
        }

        return view('admin.reports.department', array_merge($reportData, ['institution' => $institution]));
    }

    /**
     * Get institution settings for header/footer
     */
    protected function getInstitutionSettings()
    {
        return [
            'name' => Setting::get('institution.name', 'Institution Name'),
            'address' => Setting::get('institution.address', ''),
            'phone' => Setting::get('institution.phone', ''),
            'email' => Setting::get('institution.email', ''),
            'website' => Setting::get('institution.website', ''),
            'logo' => Setting::get('institution.logo', ''),
            'footer_text' => Setting::get('institution.footer_text', ''),
        ];
    }

    /**
     * Generate PDF report
     */
    protected function generatePdf($view, $data, $institution)
    {
        // For now, return HTML view. PDF generation will be implemented with dompdf/barryvdh
        // This is a placeholder - you'll need to install barryvdh/laravel-dompdf
        return view($view, array_merge($data, ['institution' => $institution, 'format' => 'pdf']));
    }

    /**
     * Generate Excel report
     */
    protected function generateExcel($data, $filename, $institution)
    {
        // Placeholder for Excel generation
        // You'll need to install maatwebsite/excel
        return response()->json(['message' => 'Excel export will be implemented']);
    }
}
