<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class SettingController extends Controller
{
    /**
     * Display settings page
     */
    public function index()
    {
        // Get all settings grouped by category
        $institutionSettings = Setting::where('key', 'like', 'institution.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        $systemSettings = Setting::where('key', 'like', 'system.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        $notificationSettings = Setting::where('key', 'like', 'notification.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        $fileSettings = Setting::where('key', 'like', 'file.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        $securitySettings = Setting::where('key', 'like', 'security.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        // Get backups
        $backups = $this->listBackups();

        return view('admin.settings.index', compact(
            'institutionSettings',
            'systemSettings',
            'notificationSettings',
            'fileSettings',
            'securitySettings',
            'backups'
        ));
    }

    /**
     * Update institution settings
     */
    public function updateInstitution(Request $request)
    {
        $request->validate([
            'institution_name' => ['required', 'string', 'max:255'],
            'institution_address' => ['nullable', 'string', 'max:500'],
            'institution_phone' => ['nullable', 'string', 'max:50'],
            'institution_email' => ['nullable', 'email', 'max:255'],
            'institution_website' => ['nullable', 'url', 'max:255'],
            'institution_logo' => ['nullable', 'image', 'mimes:jpeg,png,jpg,gif,svg', 'max:2048'],
            'institution_footer_text' => ['nullable', 'string', 'max:500'],
        ]);

        // Handle logo upload
        if ($request->hasFile('institution_logo')) {
            $file = $request->file('institution_logo');
            $path = $file->store('institution', 'public');
            
            // Delete old logo
            $oldLogo = Setting::get('institution.logo');
            if ($oldLogo && Storage::disk('public')->exists($oldLogo)) {
                Storage::disk('public')->delete($oldLogo);
            }
            
            Setting::set('institution.logo', $path, 'string', 'Institution logo path');
        }

        // Save all institution settings
        Setting::set('institution.name', $request->institution_name, 'string', 'Institution name');
        Setting::set('institution.address', $request->institution_address ?? '', 'string', 'Institution address');
        Setting::set('institution.phone', $request->institution_phone ?? '', 'string', 'Institution phone');
        Setting::set('institution.email', $request->institution_email ?? '', 'string', 'Institution email');
        Setting::set('institution.website', $request->institution_website ?? '', 'string', 'Institution website');
        Setting::set('institution.footer_text', $request->institution_footer_text ?? '', 'string', 'Footer text');

        return redirect()->route('admin.settings.index')
            ->with('success', 'Institution settings updated successfully.')
            ->with('activeTab', 'institution');
    }

    /**
     * Update system settings
     */
    public function updateSystem(Request $request)
    {
        $validated = $request->validate([
            'system_name' => ['required', 'string', 'max:255'],
            'system_timezone' => ['required', 'string', 'max:50'],
            'system_date_format' => ['required', 'string', 'max:20'],
            'system_time_format' => ['required', 'string', 'max:10'],
            'system_maintenance_mode' => ['nullable'],
            'system_maintenance_message' => ['nullable', 'string', 'max:500'],
            'system_favicon' => ['nullable', 'image', 'mimes:ico,png,jpg,jpeg,svg', 'max:1024'],
        ]);

        // Handle favicon upload
        if ($request->hasFile('system_favicon')) {
            $file = $request->file('system_favicon');
            $path = $file->store('system', 'public');
            
            // Delete old favicon
            $oldFavicon = Setting::get('system.favicon');
            if ($oldFavicon && Storage::disk('public')->exists($oldFavicon)) {
                Storage::disk('public')->delete($oldFavicon);
            }
            
            Setting::set('system.favicon', $path, 'string', 'System favicon path');
        }

        // Save system settings
        Setting::set('system.name', $validated['system_name'], 'string', 'System name');
        Setting::set('system.timezone', $validated['system_timezone'], 'string', 'System timezone');
        Setting::set('system.date_format', $validated['system_date_format'], 'string', 'Date format');
        Setting::set('system.time_format', $validated['system_time_format'], 'string', 'Time format');
        Setting::set('system.maintenance_mode', $request->has('system_maintenance_mode') ? '1' : '0', 'boolean', 'Maintenance mode');
        Setting::set('system.maintenance_message', $validated['system_maintenance_message'] ?? '', 'string', 'Maintenance message');

        return redirect()->route('admin.settings.index')
            ->with('success', 'System settings updated successfully.')
            ->with('activeTab', 'system');
    }

    /**
     * Update notification settings
     */
    public function updateNotifications(Request $request)
    {
        $validated = $request->validate([
            'notification_email_enabled' => ['nullable'],
            'notification_email_from_address' => ['nullable', 'email', 'max:255'],
            'notification_email_from_name' => ['nullable', 'string', 'max:255'],
            'notification_sms_enabled' => ['nullable'],
            'notification_submission_notify_trainer' => ['nullable'],
            'notification_submission_notify_student' => ['nullable'],
            'notification_review_notify_student' => ['nullable'],
            'notification_approval_notify_student' => ['nullable'],
        ]);

        // Save notification settings
        Setting::set('notification.email_enabled', $request->has('notification_email_enabled') ? '1' : '0', 'boolean', 'Email notifications enabled');
        Setting::set('notification.email_from_address', $validated['notification_email_from_address'] ?? '', 'string', 'Email from address');
        Setting::set('notification.email_from_name', $validated['notification_email_from_name'] ?? '', 'string', 'Email from name');
        Setting::set('notification.sms_enabled', $request->has('notification_sms_enabled') ? '1' : '0', 'boolean', 'SMS notifications enabled');
        Setting::set('notification.submission_notify_trainer', $request->has('notification_submission_notify_trainer') ? '1' : '0', 'boolean', 'Notify trainer on submission');
        Setting::set('notification.submission_notify_student', $request->has('notification_submission_notify_student') ? '1' : '0', 'boolean', 'Notify student on submission');
        Setting::set('notification.review_notify_student', $request->has('notification_review_notify_student') ? '1' : '0', 'boolean', 'Notify student on review');
        Setting::set('notification.approval_notify_student', $request->has('notification_approval_notify_student') ? '1' : '0', 'boolean', 'Notify student on approval');

        return redirect()->route('admin.settings.index')
            ->with('success', 'Notification settings updated successfully.')
            ->with('activeTab', 'notifications');
    }

    /**
     * Update file upload settings
     */
    public function updateFiles(Request $request)
    {
        $validated = $request->validate([
            'file_max_size' => ['required', 'integer', 'min:1', 'max:10240'],
            'file_allowed_types' => ['required', 'string', 'max:500'],
            'file_storage_disk' => ['required', 'string', 'max:50'],
            'file_auto_compress_images' => ['nullable'],
            'file_max_files_per_submission' => ['required', 'integer', 'min:1', 'max:100'],
        ]);

        // Save file upload settings
        Setting::set('file.max_size', (string) $validated['file_max_size'], 'integer', 'Max file size in MB');
        Setting::set('file.allowed_types', $validated['file_allowed_types'], 'string', 'Allowed file types');
        Setting::set('file.storage_disk', $validated['file_storage_disk'], 'string', 'Storage disk');
        Setting::set('file.auto_compress_images', $request->has('file_auto_compress_images') ? '1' : '0', 'boolean', 'Auto compress images');
        Setting::set('file.max_files_per_submission', (string) $validated['file_max_files_per_submission'], 'integer', 'Max files per submission');

        return redirect()->route('admin.settings.index')
            ->with('success', 'File upload settings updated successfully.')
            ->with('activeTab', 'files');
    }

    /**
     * Update security settings
     */
    public function updateSecurity(Request $request)
    {
        $validated = $request->validate([
            'security_session_timeout' => ['required', 'integer', 'min:5', 'max:1440'],
            'security_password_min_length' => ['required', 'integer', 'min:6', 'max:32'],
            'security_password_require_uppercase' => ['nullable'],
            'security_password_require_lowercase' => ['nullable'],
            'security_password_require_numbers' => ['nullable'],
            'security_password_require_symbols' => ['nullable'],
            'security_two_factor_enabled' => ['nullable'],
            'security_login_attempts_limit' => ['required', 'integer', 'min:3', 'max:10'],
            'security_login_lockout_duration' => ['required', 'integer', 'min:1', 'max:60'],
            'security_audit_log_enabled' => ['nullable'],
            'security_ip_whitelist' => ['nullable', 'string', 'max:1000'],
        ]);

        // Save security settings
        Setting::set('security.session_timeout', (string) $validated['security_session_timeout'], 'integer', 'Session timeout in minutes');
        Setting::set('security.password_min_length', (string) $validated['security_password_min_length'], 'integer', 'Password min length');
        Setting::set('security.password_require_uppercase', $request->has('security_password_require_uppercase') ? '1' : '0', 'boolean', 'Require uppercase');
        Setting::set('security.password_require_lowercase', $request->has('security_password_require_lowercase') ? '1' : '0', 'boolean', 'Require lowercase');
        Setting::set('security.password_require_numbers', $request->has('security_password_require_numbers') ? '1' : '0', 'boolean', 'Require numbers');
        Setting::set('security.password_require_symbols', $request->has('security_password_require_symbols') ? '1' : '0', 'boolean', 'Require symbols');
        Setting::set('security.two_factor_enabled', $request->has('security_two_factor_enabled') ? '1' : '0', 'boolean', 'Two-factor authentication');
        Setting::set('security.login_attempts_limit', (string) $validated['security_login_attempts_limit'], 'integer', 'Login attempts limit');
        Setting::set('security.login_lockout_duration', (string) $validated['security_login_lockout_duration'], 'integer', 'Lockout duration in minutes');
        Setting::set('security.audit_log_enabled', $request->has('security_audit_log_enabled') ? '1' : '0', 'boolean', 'Audit log enabled');
        Setting::set('security.ip_whitelist', $validated['security_ip_whitelist'] ?? '', 'string', 'IP whitelist');

        return redirect()->route('admin.settings.index')
            ->with('success', 'Security settings updated successfully.')
            ->with('activeTab', 'security');
    }

    /**
     * Delete institution logo
     */
    public function deleteLogo()
    {
        $logoPath = Setting::get('institution.logo');
        
        if ($logoPath && Storage::disk('public')->exists($logoPath)) {
            Storage::disk('public')->delete($logoPath);
        }
        
        Setting::set('institution.logo', '', 'string', 'Institution logo path');

        return redirect()->route('admin.settings.index')
            ->with('success', 'Logo deleted successfully.')
            ->with('activeTab', 'institution');
    }

    /**
     * Delete system favicon
     */
    public function deleteFavicon()
    {
        $faviconPath = Setting::get('system.favicon');
        
        if ($faviconPath && Storage::disk('public')->exists($faviconPath)) {
            Storage::disk('public')->delete($faviconPath);
        }
        
        Setting::set('system.favicon', '', 'string', 'System favicon path');

        return redirect()->route('admin.settings.index')
            ->with('success', 'Favicon deleted successfully.')
            ->with('activeTab', 'system');
    }

    /**
     * Create database backup
     */
    public function createBackup()
    {
        try {
            $connection = config('database.default');
            $database = config("database.connections.{$connection}.database");
            $username = config("database.connections.{$connection}.username");
            $password = config("database.connections.{$connection}.password");
            $host = config("database.connections.{$connection}.host");
            $port = config("database.connections.{$connection}.port", 5432);

            $timestamp = now()->format('Y-m-d_His');
            $filename = "backup_{$database}_{$timestamp}.sql";
            $backupPath = storage_path('app/backups');
            
            // Create backups directory if it doesn't exist
            if (!File::exists($backupPath)) {
                File::makeDirectory($backupPath, 0755, true);
            }

            $fullPath = $backupPath . '/' . $filename;

            // Create backup based on database type
            if ($connection === 'pgsql') {
                // PostgreSQL backup - Set environment variable in PHP for cross-platform compatibility
                $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';
                
                // Set PGPASSWORD environment variable
                putenv("PGPASSWORD={$password}");
                
                if ($isWindows) {
                    // Windows: Use cmd /c to ensure proper command execution
                    $command = sprintf(
                        'cmd /c "pg_dump -h %s -p %s -U %s -d %s -F p > %s 2> %s"',
                        escapeshellarg($host),
                        escapeshellarg($port),
                        escapeshellarg($username),
                        escapeshellarg($database),
                        escapeshellarg($fullPath),
                        escapeshellarg($fullPath . '.err')
                    );
                } else {
                    // Linux/Unix: Direct command execution
                    $command = sprintf(
                        'pg_dump -h %s -p %s -U %s -d %s -F p > %s 2> %s',
                        escapeshellarg($host),
                        escapeshellarg($port),
                        escapeshellarg($username),
                        escapeshellarg($database),
                        escapeshellarg($fullPath),
                        escapeshellarg($fullPath . '.err')
                    );
                }
                
                exec($command, $output, $returnVar);
                
                // Clear environment variable for security
                putenv('PGPASSWORD');
                
                // Check if backup file exists and has content
                if (!File::exists($fullPath) || File::size($fullPath) === 0) {
                    $errorMsg = 'Backup file was not created or is empty.';
                    if (File::exists($fullPath . '.err')) {
                        $errorContent = File::get($fullPath . '.err');
                        if (!empty(trim($errorContent))) {
                            $errorMsg .= ' Error: ' . $errorContent;
                        }
                        File::delete($fullPath . '.err');
                    }
                    throw new \Exception($errorMsg);
                }
                
                // Clean up error file if it exists (warnings are okay)
                if (File::exists($fullPath . '.err')) {
                    File::delete($fullPath . '.err');
                }
            } elseif ($connection === 'mysql') {
                // MySQL backup
                $command = sprintf(
                    'mysqldump -h %s -P %s -u %s -p%s %s > %s 2> %s',
                    escapeshellarg($host),
                    escapeshellarg($port),
                    escapeshellarg($username),
                    escapeshellarg($password),
                    escapeshellarg($database),
                    escapeshellarg($fullPath),
                    escapeshellarg($fullPath . '.err')
                );
                
                exec($command, $output, $returnVar);
                
                // Check if backup file exists and has content
                if (!File::exists($fullPath) || File::size($fullPath) === 0) {
                    $errorMsg = 'Backup file was not created or is empty.';
                    if (File::exists($fullPath . '.err')) {
                        $errorContent = File::get($fullPath . '.err');
                        if (!empty(trim($errorContent))) {
                            $errorMsg .= ' Error: ' . $errorContent;
                        }
                        File::delete($fullPath . '.err');
                    }
                    throw new \Exception($errorMsg);
                }
                
                // Clean up error file if it exists
                if (File::exists($fullPath . '.err')) {
                    File::delete($fullPath . '.err');
                }
            } else {
                throw new \Exception('Unsupported database type: ' . $connection);
            }

            // Log backup creation
            try {
                \App\Models\AuditLog::create([
                    'user_id' => auth()->id(),
                    'action' => 'backup_created',
                    'model_type' => 'system',
                    'model_id' => null,
                    'description' => "Database backup created: {$filename}",
                    'ip_address' => request()->ip(),
                    'created_at' => now(),
                ]);
            } catch (\Exception $e) {
                // Ignore audit log errors
            }

            return redirect()->route('admin.settings.index')
                ->with('success', "Database backup created successfully: {$filename}")
                ->with('activeTab', 'security');
        } catch (\Exception $e) {
            \Log::error('Backup creation failed: ' . $e->getMessage());
            return redirect()->route('admin.settings.index')
                ->with('error', 'Failed to create backup: ' . $e->getMessage())
                ->with('activeTab', 'security');
        }
    }

    /**
     * Upload and restore database backup
     */
    public function restoreBackup(Request $request)
    {
        $request->validate([
            'backup_file' => ['required', 'file', 'mimes:sql', 'max:102400'], // Max 100MB
        ]);

        try {
            $connection = config('database.default');
            $database = config("database.connections.{$connection}.database");
            $username = config("database.connections.{$connection}.username");
            $password = config("database.connections.{$connection}.password");
            $host = config("database.connections.{$connection}.host");
            $port = config("database.connections.{$connection}.port", 5432);

            $file = $request->file('backup_file');
            $tempPath = $file->storeAs('temp', 'restore_' . time() . '.sql', 'local');
            $fullPath = storage_path('app/' . $tempPath);

            // Verify file exists and has content
            if (!File::exists($fullPath) || File::size($fullPath) === 0) {
                throw new \Exception('Uploaded backup file is empty or invalid.');
            }

            // Restore backup based on database type
            if ($connection === 'pgsql') {
                // PostgreSQL restore - Set environment variable in PHP for cross-platform compatibility
                $isWindows = strtoupper(substr(PHP_OS, 0, 3)) === 'WIN';
                
                // Set PGPASSWORD environment variable
                putenv("PGPASSWORD={$password}");
                
                if ($isWindows) {
                    // Windows: Use cmd /c to ensure proper command execution
                    $command = sprintf(
                        'cmd /c "psql -h %s -p %s -U %s -d %s -f %s 2> %s"',
                        escapeshellarg($host),
                        escapeshellarg($port),
                        escapeshellarg($username),
                        escapeshellarg($database),
                        escapeshellarg($fullPath),
                        escapeshellarg($fullPath . '.err')
                    );
                } else {
                    // Linux/Unix: Direct command execution
                    $command = sprintf(
                        'psql -h %s -p %s -U %s -d %s -f %s 2> %s',
                        escapeshellarg($host),
                        escapeshellarg($port),
                        escapeshellarg($username),
                        escapeshellarg($database),
                        escapeshellarg($fullPath),
                        escapeshellarg($fullPath . '.err')
                    );
                }
                
                exec($command, $output, $returnVar);
                
                // Clear environment variable for security
                putenv('PGPASSWORD');
                
                // Check for errors
                if ($returnVar !== 0) {
                    $errorMsg = 'Restore failed.';
                    if (File::exists($fullPath . '.err')) {
                        $errorContent = File::get($fullPath . '.err');
                        // Filter out common warnings that don't indicate failure
                        if (!empty(trim($errorContent)) && !preg_match('/^(NOTICE|WARNING):/i', $errorContent)) {
                            $errorMsg .= ' Error: ' . $errorContent;
                        }
                        File::delete($fullPath . '.err');
                    }
                    File::delete($fullPath);
                    throw new \Exception($errorMsg);
                }
                
                // Clean up error file
                if (File::exists($fullPath . '.err')) {
                    File::delete($fullPath . '.err');
                }
            } elseif ($connection === 'mysql') {
                // MySQL restore
                $command = sprintf(
                    'mysql -h %s -P %s -u %s -p%s %s < %s 2> %s',
                    escapeshellarg($host),
                    escapeshellarg($port),
                    escapeshellarg($username),
                    escapeshellarg($password),
                    escapeshellarg($database),
                    escapeshellarg($fullPath),
                    escapeshellarg($fullPath . '.err')
                );
                
                exec($command, $output, $returnVar);
                
                // Check for errors
                if ($returnVar !== 0) {
                    $errorMsg = 'Restore failed.';
                    if (File::exists($fullPath . '.err')) {
                        $errorContent = File::get($fullPath . '.err');
                        if (!empty(trim($errorContent))) {
                            $errorMsg .= ' Error: ' . $errorContent;
                        }
                        File::delete($fullPath . '.err');
                    }
                    File::delete($fullPath);
                    throw new \Exception($errorMsg);
                }
                
                // Clean up error file
                if (File::exists($fullPath . '.err')) {
                    File::delete($fullPath . '.err');
                }
            } else {
                File::delete($fullPath);
                throw new \Exception('Unsupported database type: ' . $connection);
            }

            // Clean up temp file
            File::delete($fullPath);

            // Log restore action
            try {
                \App\Models\AuditLog::create([
                    'user_id' => auth()->id(),
                    'action' => 'backup_restored',
                    'model_type' => 'system',
                    'model_id' => null,
                    'description' => "Database restored from backup: {$file->getClientOriginalName()}",
                    'ip_address' => request()->ip(),
                    'created_at' => now(),
                ]);
            } catch (\Exception $e) {
                // Ignore audit log errors
            }

            return redirect()->route('admin.settings.index')
                ->with('success', "Database restored successfully from: {$file->getClientOriginalName()}")
                ->with('activeTab', 'security');
        } catch (\Exception $e) {
            \Log::error('Backup restore failed: ' . $e->getMessage());
            return redirect()->route('admin.settings.index')
                ->with('error', 'Failed to restore backup: ' . $e->getMessage())
                ->with('activeTab', 'security');
        }
    }

    /**
     * Download backup file
     */
    public function downloadBackup($filename)
    {
        $backupPath = storage_path('app/backups/' . $filename);
        
        if (!File::exists($backupPath)) {
            abort(404, 'Backup file not found');
        }

        return response()->download($backupPath, $filename);
    }

    /**
     * Delete backup file
     */
    public function deleteBackup($filename)
    {
        $backupPath = storage_path('app/backups/' . $filename);
        
        if (!File::exists($backupPath)) {
        return redirect()->route('admin.settings.index')
            ->with('error', 'Backup file not found')
            ->with('activeTab', 'security');
        }

        File::delete($backupPath);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Backup file deleted successfully')
            ->with('activeTab', 'security');
    }

    /**
     * List available backups
     */
    protected function listBackups()
    {
        $backupPath = storage_path('app/backups');
        $backups = [];

        if (File::exists($backupPath)) {
            $files = File::files($backupPath);
            foreach ($files as $file) {
                $backups[] = [
                    'filename' => $file->getFilename(),
                    'size' => $file->getSize(),
                    'created_at' => date('Y-m-d H:i:s', $file->getMTime()),
                ];
            }
            
            usort($backups, function($a, $b) {
                return strtotime($b['created_at']) - strtotime($a['created_at']);
            });
        }

        return $backups;
    }

    /**
     * Clear all data
     */
    public function clearAllData(Request $request)
    {
        $request->validate([
            'confirmation' => ['required', 'string', 'in:DELETE ALL DATA'],
        ]);

        try {
            DB::beginTransaction();

            $tables = DB::select('SELECT tablename FROM pg_tables WHERE schemaname = \'public\'');
            
            DB::statement('SET session_replication_role = replica;');

            foreach ($tables as $table) {
                $tableName = $table->tablename;
                if ($tableName !== 'migrations' && $tableName !== 'settings') {
                    DB::table($tableName)->truncate();
                }
            }

            DB::statement('SET session_replication_role = DEFAULT;');
            DB::commit();

            return redirect()->route('admin.settings.index')
                ->with('success', 'All data has been cleared successfully.')
                ->with('activeTab', 'security');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('admin.settings.index')
                ->with('error', 'Failed to clear data: ' . $e->getMessage())
                ->with('activeTab', 'security');
        }
    }
}

