<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\PoeEvidence;
use App\Models\Unit;
use App\Models\SchoolClass;
use App\Models\User;
use App\Services\ActiveTermService;
use App\Services\QueryOptimizer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class SubmissionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Use optimized query builder
        $filters = [
            'term_id' => $request->term,
            'status' => $request->status,
            'department_id' => $request->department,
            'level_id' => $request->level,
            'class_id' => $request->class,
            'unit_id' => $request->unit,
            'student_id' => $request->student,
            'search' => $request->search,
        ];

        $query = \App\Services\QueryOptimizer::submissionsQuery($filters);
        $submissions = $query->withCount('evidence')->latest('poe_submissions.created_at')->paginate(20);
        
        // Ensure we're selecting from the correct table
        $submissions->getCollection()->transform(function($submission) {
            return $submission;
        });

        // Get cached filter options
        $filterOptions = \App\Services\QueryOptimizer::getCachedFilterOptions();
        $activeTerm = ActiveTermService::getActiveTerm();

        // Optimized statistics with caching (using whereHas to avoid join conflicts)
        $stats = cache()->remember('submissions.stats.' . ($activeTerm->id ?? 'all'), 300, function() use ($activeTerm) {
            // Remove global scope to avoid JOIN conflicts with whereHas
            $baseQuery = PoeSubmission::withoutGlobalScope('activeTerm');
            
            // Use whereHas instead of join to avoid conflicts
            if ($activeTerm) {
                $baseQuery->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                });
            }
            
            return [
                'total' => (clone $baseQuery)->count(),
                'draft' => (clone $baseQuery)->where('status', 'draft')->count(),
                'submitted' => (clone $baseQuery)->where('status', 'submitted')->count(),
                'under_review' => (clone $baseQuery)->where('status', 'under_review')->count(),
                'trainer_reviewed' => (clone $baseQuery)->where('status', 'trainer_reviewed')->count(),
                'hod_approved' => (clone $baseQuery)->where('status', 'hod_approved')->count(),
                'exams_verified' => (clone $baseQuery)->where('status', 'exams_verified')->count(),
                'validator_approved' => (clone $baseQuery)->where('status', 'validator_approved')->count(),
                'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
            ];
        });

        return view('admin.submissions.index', compact(
            'submissions', 
            'activeTerm',
            'stats',
            'filterOptions'
        ));
    }

    /**
     * Display the specified resource.
     */
    public function show(PoeSubmission $submission)
    {
        // Load only what's needed for detail view
        $submission->load([
            'student:id,name,email,phone,department_id' => ['department:id,name'],
            'unit:id,name,code,department_id' => ['department:id,name'],
            'schoolClass:id,name,code,term_id,department_id,level_id' => [
                'term:id,name,status',
                'department:id,name',
                'level:id,name'
            ],
            'evidence:id,poe_submission_id,file_name,file_type,file_size,created_at',
            'reviews:id,poe_submission_id,reviewer_id,marks,feedback,status,reviewed_at' => [
                'reviewer:id,name,email'
            ],
            'validation:id,poe_submission_id,validator_id,status,comments,validated_at' => [
                'validator:id,name,email'
            ]
        ]);

        return view('admin.submissions.show', compact('submission'));
    }

    /**
     * Update submission status
     */
    public function updateStatus(Request $request, PoeSubmission $submission)
    {
        $request->validate([
            'status' => ['required', 'in:draft,submitted,under_review,trainer_reviewed,hod_approved,exams_verified,validator_approved,rejected'],
            'notes' => ['nullable', 'string'],
        ]);

        $oldStatus = $submission->status;
        $submission->update([
            'status' => $request->status,
            'notes' => $request->notes,
        ]);

        // If submitting, set submitted_at
        if ($request->status === 'submitted' && !$submission->submitted_at) {
            $submission->update(['submitted_at' => now()]);
        }

        return redirect()->back()
            ->with('success', "Submission status updated from {$oldStatus} to {$request->status}.");
    }

    /**
     * Delete submission
     */
    public function destroy(PoeSubmission $submission)
    {
        // Only allow deletion of draft submissions
        if ($submission->status !== 'draft') {
            return redirect()->back()
                ->with('error', 'Only draft submissions can be deleted.');
        }

        // Delete associated evidence files
        foreach ($submission->evidence as $evidence) {
            if (Storage::exists($evidence->file_path)) {
                Storage::delete($evidence->file_path);
            }
            $evidence->delete();
        }

        $submission->delete();

        return redirect()->route('admin.submissions.index')
            ->with('success', 'Submission deleted successfully.');
    }

    /**
     * Download evidence file
     */
    public function downloadEvidence(PoeEvidence $evidence)
    {
        if (!Storage::exists($evidence->file_path)) {
            return redirect()->back()
                ->with('error', 'File not found.');
        }

        return Storage::download($evidence->file_path, $evidence->file_name);
    }

    /**
     * Delete evidence file
     */
    public function deleteEvidence(PoeEvidence $evidence)
    {
        // Only allow deletion if submission is draft
        if ($evidence->poeSubmission->status !== 'draft') {
            return redirect()->back()
                ->with('error', 'Evidence can only be deleted from draft submissions.');
        }

        if (Storage::exists($evidence->file_path)) {
            Storage::delete($evidence->file_path);
        }

        $evidence->delete();

        return redirect()->back()
            ->with('success', 'Evidence file deleted successfully.');
    }
}
