<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Term;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TermController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $terms = Term::orderBy('start_date', 'desc')->get();
        $activeTerm = Term::getActive();
        
        return view('admin.terms.index', compact('terms', 'activeTerm'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.terms.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'start_date' => ['required', 'date', 'after_or_equal:today'],
            'end_date' => ['required', 'date', 'after:start_date'],
            'status' => ['required', 'in:draft,active'],
        ]);

        // If activating a new term, close the previous active term
        if ($request->status === 'active') {
            Term::where('status', 'active')->update(['status' => 'completed']);
            ActiveTermService::clearCache();
        }

        $term = Term::create([
            'name' => $request->name,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'status' => $request->status,
        ]);

        // Clear caches
        cache()->forget('admin.dashboard.stats');

        return redirect()->route('admin.terms.index')
            ->with('success', 'Term created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Term $term)
    {
        $term->load([
            'classes.department',
            'classes.level',
            'termDepartmentUnits.unit',
            'termDepartmentUnits.department'
        ]);

        // Calculate accurate statistics matching actual data
        $stats = [
            // Total Classes: Count classes belonging to this term
            'total_classes' => $term->classes()->count(),
            
            // Total Units: Count unique units from term_department_units table
            // This reflects units configured/available for this term
            'total_units' => (int) DB::table('term_department_units')
                ->where('term_id', $term->id)
                ->selectRaw('COUNT(DISTINCT unit_id) as count')
                ->value('count') ?? 0,
            
            // Total Students: Count distinct active students enrolled in classes of this term
            'total_students' => (int) DB::table('enrollments')
                ->join('classes', 'enrollments.class_id', '=', 'classes.id')
                ->where('classes.term_id', $term->id)
                ->where('enrollments.status', 'active')
                ->selectRaw('COUNT(DISTINCT enrollments.student_id) as count')
                ->value('count') ?? 0,
            
            // Total Submissions: Count PoE submissions for classes in this term
            'total_submissions' => DB::table('poe_submissions')
                ->join('classes', 'poe_submissions.class_id', '=', 'classes.id')
                ->where('classes.term_id', $term->id)
                ->count(),
        ];

        return view('admin.terms.show', compact('term', 'stats'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Term $term)
    {
        // Prevent editing closed terms
        if ($term->isClosed()) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Closed terms cannot be edited.');
        }

        return view('admin.terms.edit', compact('term'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Term $term)
    {
        // Prevent editing closed terms
        if ($term->isClosed()) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Closed terms cannot be edited.');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'start_date' => ['required', 'date'],
            'end_date' => ['required', 'date', 'after:start_date'],
            'status' => ['required', 'in:draft,active,completed'],
        ]);

        // If activating this term, close the previous active term
        if ($request->status === 'active' && !$term->isActive()) {
            Term::where('status', 'active')
                ->where('id', '!=', $term->id)
                ->update(['status' => 'completed']);
            ActiveTermService::clearCache();
        }

        $term->update([
            'name' => $request->name,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'status' => $request->status,
        ]);

        // Clear caches
        cache()->forget('admin.dashboard.stats');

        return redirect()->route('admin.terms.index')
            ->with('success', 'Term updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Term $term)
    {
        // Prevent deletion of active or closed terms
        if ($term->isActive()) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Active terms cannot be deleted.');
        }

        if ($term->isClosed()) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Closed terms cannot be deleted for audit purposes.');
        }

        // Check if term has associated data
        if ($term->classes()->count() > 0 || $term->termDepartmentUnits()->count() > 0) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Cannot delete term with associated classes or units.');
        }

        $term->delete();

        // Clear dashboard cache
        cache()->forget('admin.dashboard.stats');

        return redirect()->route('admin.terms.index')
            ->with('success', 'Term deleted successfully.');
    }

    /**
     * Activate a term (closes previous active term)
     */
    public function activate(Term $term)
    {
        if ($term->isClosed()) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Closed terms cannot be activated.');
        }

        $term->activate();

        // Clear caches
        ActiveTermService::clearCache();
        cache()->forget('admin.dashboard.stats');

        return redirect()->route('admin.terms.index')
            ->with('success', 'Term activated successfully. Previous active term has been closed.');
    }

    /**
     * Close a term
     */
    public function close(Term $term)
    {
        if (!$term->isActive()) {
            return redirect()->route('admin.terms.index')
                ->with('error', 'Only active terms can be closed.');
        }

        $term->close();

        // Clear all active term caches
        ActiveTermService::clearCache();
        
        // Clear dashboard and statistics caches
        cache()->forget('admin.dashboard.stats');
        cache()->forget('hod.dashboard.stats');
        cache()->forget('trainer.dashboard.stats');
        
        // Clear Redis caches if using Redis
        if (config('cache.default') === 'redis') {
            cache()->flush(); // Clear all cached data for clean workspace
        }

        return redirect()->route('admin.terms.index')
            ->with('success', 'Term closed successfully. All portals have been reset to clean workspace.');
    }
}
