<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Unit;
use App\Models\Department;
use App\Models\SchoolClass;
use Illuminate\Http\Request;

class UnitController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Unit::with('department')->withCount('classes');

        // Filter by department if provided
        if ($request->has('department') && $request->department) {
            $query->where('department_id', $request->department);
        }

        $units = $query->orderBy('name')->paginate(15);
        $departments = Department::orderBy('name')->get();

        return view('admin.units.index', compact('units', 'departments'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $departments = Department::orderBy('name')->get();
        return view('admin.units.create', compact('departments'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:units,code'],
            'description' => ['nullable', 'string'],
            'department_id' => ['required', 'exists:departments,id'],
        ]);

        Unit::create($request->only(['name', 'code', 'description', 'department_id']));

        return redirect()->route('admin.units.index')
            ->with('success', 'Unit created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Unit $unit)
    {
        $unit->load(['department', 'classes.term', 'classes.department', 'classes.level', 'trainers.department']);
        $allClasses = SchoolClass::with(['term', 'department', 'level'])
            ->where('department_id', $unit->department_id) // Only show classes from same department
            ->orderBy('name')
            ->get();
        
        // Get all active trainers from the same department as the unit
        $availableTrainers = \App\Models\User::where('role', 'trainer')
            ->where('department_id', $unit->department_id)
            ->where('status', 'active')
            ->orderBy('name')
            ->get();

        return view('admin.units.show', compact('unit', 'allClasses', 'availableTrainers'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Unit $unit)
    {
        $departments = Department::orderBy('name')->get();
        return view('admin.units.edit', compact('unit', 'departments'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Unit $unit)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:units,code,' . $unit->id],
            'description' => ['nullable', 'string'],
            'department_id' => ['required', 'exists:departments,id'],
        ]);

        $unit->update($request->only(['name', 'code', 'description', 'department_id']));

        return redirect()->route('admin.units.index')
            ->with('success', 'Unit updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Unit $unit)
    {
        // Check if unit has classes or submissions
        if ($unit->classes()->count() > 0 || $unit->poeSubmissions()->count() > 0) {
            return redirect()->route('admin.units.index')
                ->with('error', 'Cannot delete unit with associated classes or submissions.');
        }

        $unit->delete();

        return redirect()->route('admin.units.index')
            ->with('success', 'Unit deleted successfully.');
    }

    /**
     * Assign unit to class
     */
    public function assignToClass(Request $request, Unit $unit)
    {
        $request->validate([
            'class_id' => ['required', 'exists:classes,id'],
        ]);

        // Check if already assigned
        if ($unit->classes()->where('class_id', $request->class_id)->exists()) {
            return redirect()->back()
                ->with('error', 'This unit is already assigned to this class.');
        }

        $unit->classes()->attach($request->class_id);

        return redirect()->back()
            ->with('success', 'Unit assigned to class successfully.');
    }

    /**
     * Remove unit from class
     */
    public function removeFromClass(Request $request, Unit $unit)
    {
        $request->validate([
            'class_id' => ['required', 'exists:classes,id'],
        ]);

        // Check if there are submissions for this unit in this class
        $hasSubmissions = \App\Models\PoeSubmission::where('unit_id', $unit->id)
            ->where('class_id', $request->class_id)
            ->exists();

        if ($hasSubmissions) {
            return redirect()->back()
                ->with('error', 'Cannot remove unit from class with associated submissions.');
        }

        $unit->classes()->detach($request->class_id);

        return redirect()->back()
            ->with('success', 'Unit removed from class successfully.');
    }

    /**
     * Assign trainer to unit
     */
    public function assignTrainer(Request $request, Unit $unit)
    {
        $request->validate([
            'trainer_id' => ['required', 'exists:users,id'],
        ]);

        $trainer = \App\Models\User::findOrFail($request->trainer_id);

        // Verify trainer role
        if (!$trainer->isTrainer()) {
            return redirect()->back()
                ->with('error', 'Selected user is not a trainer.');
        }

        // Verify trainer is in the same department as the unit
        if ($trainer->department_id !== $unit->department_id) {
            return redirect()->back()
                ->with('error', 'Trainer must be in the same department as the unit.');
        }

        // Check if already assigned
        if ($unit->trainers()->where('trainer_id', $request->trainer_id)->exists()) {
            return redirect()->back()
                ->with('error', 'This trainer is already assigned to this unit.');
        }

        $unit->trainers()->attach($request->trainer_id);

        // Log the action
        \App\Models\AuditLog::log('trainer_assigned', $unit, [
            'trainer_id' => $trainer->id,
            'trainer_name' => $trainer->name,
            'unit_id' => $unit->id,
            'unit_name' => $unit->name,
        ]);

        return redirect()->back()
            ->with('success', 'Trainer assigned to unit successfully.');
    }

    /**
     * Remove trainer from unit
     */
    public function removeTrainer(Request $request, Unit $unit)
    {
        $request->validate([
            'trainer_id' => ['required', 'exists:users,id'],
        ]);

        $trainer = \App\Models\User::findOrFail($request->trainer_id);

        // Check if trainer has pending reviews for this unit
        $hasPendingReviews = \App\Models\Review::where('reviewer_id', $trainer->id)
            ->whereHas('poeSubmission', function($query) use ($unit) {
                $query->where('unit_id', $unit->id)
                      ->whereIn('status', ['submitted', 'under_review']);
            })
            ->exists();

        if ($hasPendingReviews) {
            return redirect()->back()
                ->with('error', 'Cannot remove trainer with pending reviews for this unit.');
        }

        $unit->trainers()->detach($request->trainer_id);

        // Log the action
        \App\Models\AuditLog::log('trainer_removed', $unit, [
            'trainer_id' => $trainer->id,
            'trainer_name' => $trainer->name,
            'unit_id' => $unit->id,
            'unit_name' => $unit->name,
        ]);

        return redirect()->back()
            ->with('success', 'Trainer removed from unit successfully.');
    }
}
