<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Department;
use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = User::query();

        // Filter by role if provided
        if ($request->has('role') && $request->role) {
            $query->where('role', $request->role);
        }

        // Filter by status if provided
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        // Filter by department if provided
        if ($request->has('department') && $request->department) {
            $query->where('department_id', $request->department);
        }

        // Search by name or email
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        $users = $query->with('department:id,name')->latest()->paginate(15);
        $departments = Department::select('id', 'name')->orderBy('name')->get();
        $selectedRole = $request->role ?? 'all';
        $selectedStatus = $request->status ?? 'all';
        $selectedDepartment = $request->department ?? 'all';
        $search = $request->search ?? '';

        // Get counts by role - single optimized query
        $roleCounts = cache()->remember('admin.users.role_counts', 300, function () {
            return User::selectRaw('role, count(*) as count')
                ->groupBy('role')
                ->get()
                ->pluck('count', 'role')
                ->toArray();
        });
        
        // Ensure all roles are present
        $allRoles = ['admin', 'exams_office', 'hod', 'trainer', 'validator', 'student'];
        foreach ($allRoles as $role) {
            if (!isset($roleCounts[$role])) {
                $roleCounts[$role] = 0;
            }
        }

        return view('admin.users.index', compact(
            'users', 
            'departments', 
            'selectedRole', 
            'selectedStatus', 
            'selectedDepartment',
            'search',
            'roleCounts'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $departments = Department::select('id', 'name')->orderBy('name')->get();
        $units = Unit::select('id', 'name', 'code', 'department_id')->orderBy('name')->get();
        return view('admin.users.create', compact('departments', 'units'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'in:admin,hod,trainer,exams_office,student,validator'],
            'department_id' => ['nullable', 'exists:departments,id'],
            'phone' => ['nullable', 'string', 'max:20'],
            'status' => ['required', 'in:active,inactive,suspended'],
            'unit_ids' => ['nullable', 'array'],
            'unit_ids.*' => ['exists:units,id'],
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'department_id' => $request->department_id,
            'phone' => $request->phone,
            'status' => $request->status,
            'email_verified_at' => now(),
        ]);

        // If trainer, assign units only if provided
        if ($user->isTrainer() && $user->department_id && $request->has('unit_ids') && !empty($request->unit_ids)) {
            // Assign specific units if provided
            $unitIds = $request->unit_ids;
            // Validate that all units belong to the trainer's department
            $validUnitIds = Unit::where('department_id', $user->department_id)
                ->whereIn('id', $unitIds)
                ->pluck('id')
                ->toArray();
            if (!empty($validUnitIds)) {
                $user->units()->attach($validUnitIds);
            }
        }

        // Clear cache when user is created
        cache()->forget('admin.dashboard.stats');
        cache()->forget('admin.dashboard.users_by_role');
        cache()->forget('admin.users.role_counts');

        return redirect()->route('admin.users.index', ['role' => $request->role])
            ->with('success', 'User created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(User $user)
    {
        $user->load([
            'department:id,name',
            'enrollments' => function($query) {
                $query->select('id', 'student_id', 'class_id', 'status', 'enrolled_at')
                      ->with(['schoolClass:id,name,term_id,department_id,level_id' => function($q) {
                          $q->with(['term:id,name', 'department:id,name', 'level:id,name']);
                      }]);
            },
            'poeSubmissions' => function($query) {
                $query->select('id', 'student_id', 'unit_id', 'class_id', 'status', 'version', 'submitted_at')
                      ->with(['unit:id,name', 'schoolClass:id,name']);
            }
        ]);
        return view('admin.users.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        $departments = Department::select('id', 'name')->orderBy('name')->get();
        $units = Unit::select('id', 'name', 'code', 'department_id')->orderBy('name')->get();
        $user->load('units:id');
        $assignedUnitIds = $user->units->pluck('id')->toArray();
        return view('admin.users.edit', compact('user', 'departments', 'units', 'assignedUnitIds'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'in:admin,hod,trainer,exams_office,student,validator'],
            'department_id' => ['nullable', 'exists:departments,id'],
            'phone' => ['nullable', 'string', 'max:20'],
            'status' => ['required', 'in:active,inactive,suspended'],
            'unit_ids' => ['nullable', 'array'],
            'unit_ids.*' => ['exists:units,id'],
        ]);

        $oldDepartmentId = $user->department_id;
        $wasTrainer = $user->isTrainer();

        $data = [
            'name' => $request->name,
            'email' => $request->email,
            'role' => $request->role,
            'department_id' => $request->department_id,
            'phone' => $request->phone,
            'status' => $request->status,
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);

        // Handle unit assignments for trainers
        if ($user->isTrainer() && $user->department_id) {
            if ($request->has('unit_ids')) {
                // Assign specific units if provided
                $unitIds = $request->unit_ids;
                // Validate that all units belong to the trainer's department
                $validUnitIds = Unit::where('department_id', $user->department_id)
                    ->whereIn('id', $unitIds)
                    ->pluck('id')
                    ->toArray();
                
                // Sync units (removes old, adds new) - can be empty array
                $user->units()->sync($validUnitIds);
            } else {
                // If no units selected, clear all assignments
                $user->units()->sync([]);
            }
        } else {
            // If user is no longer a trainer, remove all unit assignments
            if ($wasTrainer && !$user->isTrainer()) {
                $user->units()->detach();
            }
        }

        // Clear cache when user is updated
        cache()->forget('admin.dashboard.stats');
        cache()->forget('admin.dashboard.users_by_role');
        cache()->forget('admin.users.role_counts');

        return redirect()->route('admin.users.index', ['role' => $request->role])
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return redirect()->route('admin.users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        // Clear cache when user is deleted
        cache()->forget('admin.dashboard.stats');
        cache()->forget('admin.dashboard.users_by_role');
        cache()->forget('admin.users.role_counts');

        return redirect()->route('admin.users.index')
            ->with('success', 'User deleted successfully.');
    }
}
