<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Illuminate\View\View;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        // Get departments
        $departments = \App\Models\Department::select('id', 'name', 'code')
            ->orderBy('name')
            ->get();

        // Get active classes for the current term with department info
        $activeTermId = \App\Services\ActiveTermService::getActiveTermId();
        $classes = \App\Models\SchoolClass::select('id', 'name', 'code', 'department_id', 'level_id')
            ->when($activeTermId, function($query) use ($activeTermId) {
                $query->where('term_id', $activeTermId);
            })
            ->with(['department:id,name', 'level:id,name'])
            ->orderBy('name')
            ->get();

        return view('auth.register', compact('departments', 'classes'));
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'admission_number' => ['required', 'string', 'max:255', 'unique:users,admission_number'],
            'department_id' => ['required', 'exists:departments,id'],
            'class_id' => ['required', 'exists:classes,id'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['nullable', 'in:student'], // Only allow student registration
        ]);

        // Verify that the selected class belongs to the selected department
        $class = \App\Models\SchoolClass::findOrFail($request->class_id);
        if ($class->department_id != $request->department_id) {
            return back()
                ->withInput()
                ->withErrors(['class_id' => 'The selected class does not belong to the selected department.']);
        }

        // Combine first and last name for the name field
        $fullName = trim($request->first_name . ' ' . $request->last_name);

        $user = User::create([
            'name' => $fullName,
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'admission_number' => $request->admission_number,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => 'student',
            'department_id' => $request->department_id,
            'status' => 'active', // New registrations are active by default
        ]);

        // Create enrollment for the student
        \App\Models\Enrollment::create([
            'student_id' => $user->id,
            'class_id' => $request->class_id,
            'enrolled_at' => now(),
            'status' => 'active',
        ]);

        event(new Registered($user));

        Auth::login($user);

        return redirect(RouteServiceProvider::HOME);
    }
}
