<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\Unit;
use App\Notifications\GeneralNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class AssignmentController extends Controller
{
    /**
     * Authorize: HOD can only manage assignments they created (trainer_id = HOD) for units in their department.
     */
    private function authorizeAssignment(Assignment $assignment): void
    {
        $hod = Auth::user();
        if ($assignment->trainer_id !== $hod->id) {
            abort(403, 'You do not have permission to manage this assignment.');
        }
        $assignment->load('unit:id,department_id');
        if ($assignment->unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to manage this assignment.');
        }
    }

    /**
     * Display the specified assignment.
     */
    public function show(Assignment $assignment)
    {
        $this->authorizeAssignment($assignment);

        $assignment->load(['unit:id,name,code', 'trainer:id,name']);

        $submissionsCount = $assignment->submissions()->count();
        $submittedCount = $assignment->submissions()->where('status', 'submitted')->count();
        $gradedCount = $assignment->submissions()->whereNotNull('marks')->count();

        $submissions = $assignment->submissions()
            ->with(['student:id,name,email,admission_number'])
            ->orderBy('submitted_at', 'desc')
            ->get();

        return view('hod.assignments.show', compact('assignment', 'submissionsCount', 'submittedCount', 'gradedCount', 'submissions'));
    }

    /**
     * Show the form for editing the specified assignment.
     */
    public function edit(Assignment $assignment)
    {
        $this->authorizeAssignment($assignment);

        $assignment->load('unit:id,name,code');

        return view('hod.assignments.edit', compact('assignment'));
    }

    /**
     * Store a newly created assignment for a unit (HOD can create assignments, exams, practicals, projects).
     */
    public function store(Request $request, Unit $unit)
    {
        $hod = Auth::user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return back()->with('error', 'No active term found.');
        }

        if ($unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to create assignments for this unit.');
        }

        $departmentClassIds = \App\Models\SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        $assignedClassIds = $unit->classes()
            ->whereIn('classes.id', $departmentClassIds)
            ->pluck('classes.id')
            ->toArray();

        if (empty($assignedClassIds)) {
            return back()->with('error', 'No classes are assigned to this unit in the current term.');
        }

        if (!$unit->isConfigured()) {
            return back()->with('error', 'This unit has not been configured. Please configure the assessment structure first.');
        }

        $rules = [
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:assignment,exam,practical,project',
            'class_id' => ['required', 'exists:classes,id', function ($attribute, $value, $fail) use ($assignedClassIds) {
                if (!in_array((int) $value, $assignedClassIds)) {
                    $fail('The selected class is not assigned to this unit.');
                }
            }],
            'total_marks' => 'nullable|integer|min:0',
            'due_date' => 'nullable|date',
            'start_date' => 'nullable|date',
            'instructions' => 'nullable|string',
            'is_published' => 'boolean',
        ];

        $messages = [];
        if ($request->type === 'practical') {
            $rules['candidate_tool'] = 'required|array|min:1';
            $rules['assessor_tool'] = 'required|array|min:1';
            $rules['candidate_tool.*'] = 'file|max:10240';
            $rules['assessor_tool.*'] = 'file|max:10240';
            $messages['candidate_tool.required'] = 'Candidate Tool is required for practical assignments.';
            $messages['assessor_tool.required'] = 'Assessor Tool is required for practical assignments.';
        } else {
            $rules['attachments'] = 'nullable|array';
            $rules['attachments.*'] = 'file|max:10240';
            $rules['candidate_tool'] = 'nullable|array';
            $rules['assessor_tool'] = 'nullable|array';
        }

        $validated = $request->validate($rules, $messages);

        $attachmentPaths = [];
        $candidateToolPaths = [];
        $assessorToolPaths = [];

        if ($validated['type'] === 'practical') {
            $sanitizedTitle = $this->sanitizeFilename($validated['title']);
            if ($request->hasFile('candidate_tool')) {
                $index = 0;
                foreach ($request->file('candidate_tool') as $file) {
                    $ext = $file->getClientOriginalExtension();
                    $filename = 'candidate_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $ext;
                    $path = $file->storeAs('assignments/' . $unit->id . '/candidate-tool', $filename, 'public');
                    $candidateToolPaths[] = $path;
                    $index++;
                }
            }
            if ($request->hasFile('assessor_tool')) {
                $index = 0;
                foreach ($request->file('assessor_tool') as $file) {
                    $ext = $file->getClientOriginalExtension();
                    $filename = 'assessor_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $ext;
                    $path = $file->storeAs('assignments/' . $unit->id . '/assessor-tool', $filename, 'public');
                    $assessorToolPaths[] = $path;
                    $index++;
                }
            }
        } else {
            if ($request->hasFile('attachments')) {
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('assignments/' . $unit->id, 'public');
                    $attachmentPaths[] = $path;
                }
            }
        }

        $assignment = Assignment::create([
            'unit_id' => $unit->id,
            'trainer_id' => $hod->id,
            'class_id' => $validated['class_id'],
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'type' => $validated['type'],
            'total_marks' => $validated['total_marks'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'start_date' => $validated['start_date'] ?? null,
            'instructions' => $validated['instructions'] ?? null,
            'is_published' => !empty($validated['is_published']),
            'attachments' => !empty($attachmentPaths) ? $attachmentPaths : null,
            'candidate_tool' => !empty($candidateToolPaths) ? $candidateToolPaths : null,
            'assessor_tool' => !empty($assessorToolPaths) ? $assessorToolPaths : null,
        ]);

        if ($assignment->is_published) {
            $this->notifyStudentsInClass($assignment, $hod, 'New assignment: ' . $assignment->title);
        }

        return redirect()->route('hod.units.show', $unit)
            ->with('success', 'Assignment created successfully.');
    }

    /**
     * Update the specified assignment.
     */
    public function update(Request $request, Assignment $assignment)
    {
        $this->authorizeAssignment($assignment);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'type' => 'required|in:assignment,exam,practical,project',
            'total_marks' => 'nullable|integer|min:0',
            'due_date' => 'nullable|date',
            'start_date' => 'nullable|date',
            'instructions' => 'nullable|string',
            'is_published' => 'boolean',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240',
            'candidate_tool' => 'nullable|array',
            'candidate_tool.*' => 'file|max:10240',
            'assessor_tool' => 'nullable|array',
            'assessor_tool.*' => 'file|max:10240',
        ]);

        // On edit, new files REPLACE existing ones.
        $attachmentPaths = $assignment->attachments ?? [];
        $candidateToolPaths = $assignment->candidate_tool ?? [];
        $assessorToolPaths = $assignment->assessor_tool ?? [];

        if ($validated['type'] === 'practical') {
            $sanitizedTitle = $this->sanitizeFilename($validated['title']);
            if ($request->hasFile('candidate_tool')) {
                foreach ($candidateToolPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
                $candidateToolPaths = [];
                $index = 0;
                foreach ($request->file('candidate_tool') as $file) {
                    $ext = $file->getClientOriginalExtension();
                    $filename = 'candidate_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $ext;
                    $path = $file->storeAs('assignments/' . $assignment->unit_id . '/candidate-tool', $filename, 'public');
                    $candidateToolPaths[] = $path;
                    $index++;
                }
            }
            if ($request->hasFile('assessor_tool')) {
                foreach ($assessorToolPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
                $assessorToolPaths = [];
                $index = 0;
                foreach ($request->file('assessor_tool') as $file) {
                    $ext = $file->getClientOriginalExtension();
                    $filename = 'assessor_tool_' . $sanitizedTitle . ($index > 0 ? '_' . ($index + 1) : '') . '.' . $ext;
                    $path = $file->storeAs('assignments/' . $assignment->unit_id . '/assessor-tool', $filename, 'public');
                    $assessorToolPaths[] = $path;
                    $index++;
                }
            }
        } else {
            if ($request->hasFile('attachments')) {
                foreach ($attachmentPaths as $path) {
                    Storage::disk('public')->delete($path);
                }
                $attachmentPaths = [];
                foreach ($request->file('attachments') as $file) {
                    $path = $file->store('assignments/' . $assignment->unit_id, 'public');
                    $attachmentPaths[] = $path;
                }
            }
        }

        $wasPublished = $assignment->is_published;
        $assignment->update([
            'title' => $validated['title'],
            'description' => $validated['description'] ?? null,
            'type' => $validated['type'],
            'total_marks' => $validated['total_marks'] ?? null,
            'due_date' => $validated['due_date'] ?? null,
            'start_date' => $validated['start_date'] ?? null,
            'instructions' => $validated['instructions'] ?? null,
            'is_published' => !empty($validated['is_published']),
            'attachments' => !empty($attachmentPaths) ? $attachmentPaths : null,
            'candidate_tool' => !empty($candidateToolPaths) ? $candidateToolPaths : null,
            'assessor_tool' => !empty($assessorToolPaths) ? $assessorToolPaths : null,
        ]);

        if ($assignment->is_published && !$wasPublished) {
            $this->notifyStudentsInClass($assignment, Auth::user(), 'New assignment: ' . $assignment->title);
        }

        return redirect()->to(route('hod.units.show', $assignment->unit) . (request('context') === 'departmental' ? '?context=departmental' : ''))
            ->with('success', 'Assignment updated successfully.');
    }

    /**
     * Remove the specified assignment.
     */
    public function destroy(Assignment $assignment)
    {
        $this->authorizeAssignment($assignment);

        if ($assignment->attachments) {
            foreach ($assignment->attachments as $path) {
                Storage::disk('public')->delete($path);
            }
        }
        if ($assignment->candidate_tool) {
            foreach ($assignment->candidate_tool as $path) {
                Storage::disk('public')->delete($path);
            }
        }
        if ($assignment->assessor_tool) {
            foreach ($assignment->assessor_tool as $path) {
                Storage::disk('public')->delete($path);
            }
        }

        $unit = $assignment->unit;
        $assignment->delete();

        return redirect()->to(route('hod.units.show', $unit) . (request('context') === 'departmental' ? '?context=departmental' : ''))
            ->with('success', 'Assignment deleted successfully.');
    }

    /**
     * Download assignment attachment.
     */
    public function downloadAttachment(Assignment $assignment, $index)
    {
        $this->authorizeAssignment($assignment);

        if (!$assignment->attachments || !isset($assignment->attachments[$index])) {
            abort(404, 'File not found.');
        }

        $filePath = $assignment->attachments[$index];
        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('public')->download($filePath);
    }

    /**
     * Download practical tool (candidate or assessor).
     */
    public function downloadTool(Assignment $assignment, string $toolType, $index)
    {
        $this->authorizeAssignment($assignment);

        if ($assignment->type !== 'practical') {
            abort(404, 'This assignment does not have tools.');
        }

        $fileArray = null;
        if ($toolType === 'candidate') {
            $fileArray = $assignment->candidate_tool;
        } elseif ($toolType === 'assessor') {
            $fileArray = $assignment->assessor_tool;
        } else {
            abort(404, 'Invalid tool type.');
        }

        if (!$fileArray || !isset($fileArray[$index])) {
            abort(404, 'File not found.');
        }

        $filePath = $fileArray[$index];
        if (!Storage::disk('public')->exists($filePath)) {
            abort(404, 'File not found.');
        }

        return Storage::disk('public')->download($filePath);
    }

    /**
     * Show the form for marking a submission (non-practical).
     */
    public function showSubmission(Assignment $assignment, AssignmentSubmission $submission)
    {
        $this->authorizeAssignment($assignment);

        if ($submission->assignment_id !== $assignment->id) {
            abort(404, 'Submission not found for this assignment.');
        }

        if ($assignment->type === 'practical') {
            abort(403, 'Practical assignments use the Mark Assessor Tool interface.');
        }

        $submission->load(['student:id,name,email,admission_number', 'assignment:id,title,type,total_marks']);

        return view('hod.assignments.mark-submission', compact('assignment', 'submission'));
    }

    /**
     * Update marks for a submission (non-practical).
     */
    public function markSubmission(Request $request, Assignment $assignment, AssignmentSubmission $submission)
    {
        $this->authorizeAssignment($assignment);

        if ($submission->assignment_id !== $assignment->id) {
            abort(404, 'Submission not found for this assignment.');
        }

        if ($assignment->type === 'practical') {
            abort(403, 'Practical assignments use the Mark Assessor Tool interface.');
        }

        $maxMarks = $assignment->total_marks ?? 100;

        $validated = $request->validate([
            'marks' => ['required', 'numeric', 'min:0', 'max:' . $maxMarks],
            'trainer_feedback' => ['nullable', 'string', 'max:5000'],
        ]);

        $submission->update([
            'marks' => $validated['marks'],
            'trainer_feedback' => $validated['trainer_feedback'] ?? null,
            'status' => 'graded',
            'graded_at' => now(),
        ]);

        $redirectUrl = route('hod.assignments.show', $assignment) . (request('context') === 'departmental' ? '?context=departmental' : '');
        return redirect()->to($redirectUrl)->with('success', 'Submission marked successfully.');
    }

    /**
     * Notify all students enrolled in the assignment's class about the assignment.
     */
    private function notifyStudentsInClass(Assignment $assignment, $sender, string $message): void
    {
        $assignment->load('unit:id,name');
        $studentIds = DB::table('enrollments')
            ->where('class_id', $assignment->class_id)
            ->where('status', 'active')
            ->pluck('student_id');

        $students = \App\Models\User::whereIn('id', $studentIds)
            ->where('role', 'student')
            ->where('status', 'active')
            ->get();

        foreach ($students as $student) {
            $student->notify(new GeneralNotification(
                'New assignment',
                $message . ' — Unit: ' . ($assignment->unit->name ?? '') . '. Check your unit page to view and submit.',
                $sender->id
            ));
        }
    }

    private function sanitizeFilename($title)
    {
        $s = strtolower(preg_replace('/[^a-z0-9]+/', '_', trim($title)));
        $s = substr(trim($s, '_'), 0, 50);
        return $s ?: 'assignment';
    }
}
