<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\SchoolClass;
use App\Models\Term;
use App\Models\Level;
use App\Models\Unit;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ClassController extends Controller
{
    /**
     * Display a listing of classes in HOD's department.
     */
    public function index(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$hod->department_id) {
            return view('hod.classes.index', [
                'classes' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('warning', 'You are not assigned to a department.');
        }

        $query = SchoolClass::where('department_id', $hod->department_id)
            ->with(['term:id,name', 'level:id,name']);

        // Filter by term if provided (non-empty value means specific term selected)
        if ($request->filled('term')) {
            $query->where('term_id', $request->term);
        } elseif ($activeTerm) {
            // Default to active term when no filter is selected
            $query->where('term_id', $activeTerm->id);
        }

        // Filter by level if provided
        if ($request->has('level') && $request->level) {
            $query->where('level_id', $request->level);
        }

        // Search
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('code', 'like', "%{$search}%");
            });
        }

        $classes = $query->withCount(['enrollments', 'units'])->latest()->paginate(15);
        $terms = Term::orderBy('start_date', 'desc')->get();
        $levels = Level::orderBy('name')->get();

        return view('hod.classes.index', compact('classes', 'terms', 'levels', 'activeTerm'));
    }

    /**
     * Show the form for creating a new class.
     */
    public function create()
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$hod->department_id) {
            return redirect()->route('hod.classes.index')
                ->with('warning', 'You are not assigned to a department.');
        }

        $levels = Level::orderBy('name')->get();
        $terms = Term::orderBy('start_date', 'desc')->get();

        return view('hod.classes.create', compact('levels', 'terms', 'activeTerm'));
    }

    /**
     * Store a newly created class.
     */
    public function store(Request $request)
    {
        $hod = Auth::user();

        if (!$hod->department_id) {
            return redirect()->back()
                ->with('error', 'You are not assigned to a department.');
        }

        $request->validate([
            'term_id' => ['required', 'exists:terms,id'],
            'level_id' => ['required', 'exists:levels,id'],
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:classes,code'],
        ]);

        // Verify term exists and is accessible
        $term = Term::findOrFail($request->term_id);

        SchoolClass::create([
            'term_id' => $request->term_id,
            'department_id' => $hod->department_id, // Always set to HOD's department
            'level_id' => $request->level_id,
            'name' => $request->name,
            'code' => $request->code,
        ]);

        return redirect()->route('hod.classes.index')
            ->with('success', 'Class created successfully.');
    }

    /**
     * Display the specified class.
     */
    public function show(SchoolClass $class)
    {
        $hod = Auth::user();

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this class.');
        }

        $class->load([
            'term:id,name,status',
            'department:id,name',
            'level:id,name',
            'enrollments:id,student_id,class_id,status' => [
                'student:id,name,email'
            ],
            'units:id,name,code,department_id'
        ]);

        $stats = [
            'total_students' => $class->enrollments()->count(),
            'active_students' => $class->enrollments()->where('status', 'active')->count(),
            'total_submissions' => $class->poeSubmissions()->count(),
            'total_units' => $class->units()->count(),
        ];

        // Get available units from department that aren't already assigned
        $assignedUnitIds = $class->units->pluck('id')->toArray();
        $availableUnits = Unit::select('id', 'name', 'code')
            ->where('department_id', $hod->department_id)
            ->whereNotIn('id', $assignedUnitIds)
            ->orderBy('name')
            ->get();

        return view('hod.classes.show', compact('class', 'stats', 'availableUnits'));
    }

    /**
     * Show the form for editing the specified class.
     */
    public function edit(SchoolClass $class)
    {
        $hod = Auth::user();

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to edit this class.');
        }

        $levels = Level::orderBy('name')->get();
        $terms = Term::orderBy('start_date', 'desc')->get();

        return view('hod.classes.edit', compact('class', 'levels', 'terms'));
    }

    /**
     * Update the specified class.
     */
    public function update(Request $request, SchoolClass $class)
    {
        $hod = Auth::user();

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to update this class.');
        }

        $request->validate([
            'term_id' => ['required', 'exists:terms,id'],
            'level_id' => ['required', 'exists:levels,id'],
            'name' => ['required', 'string', 'max:255'],
            'code' => ['required', 'string', 'max:50', 'unique:classes,code,' . $class->id],
        ]);

        $class->update([
            'term_id' => $request->term_id,
            'department_id' => $hod->department_id, // Keep in HOD's department
            'level_id' => $request->level_id,
            'name' => $request->name,
            'code' => $request->code,
        ]);

        return redirect()->route('hod.classes.index')
            ->with('success', 'Class updated successfully.');
    }

    /**
     * Remove the specified class.
     */
    public function destroy(SchoolClass $class)
    {
        $hod = Auth::user();

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to delete this class.');
        }

        // Check if class has enrollments or submissions
        if ($class->enrollments()->count() > 0 || $class->poeSubmissions()->count() > 0) {
            return redirect()->route('hod.classes.index')
                ->with('error', 'Cannot delete class with associated enrollments or submissions.');
        }

        $class->delete();

        return redirect()->route('hod.classes.index')
            ->with('success', 'Class deleted successfully.');
    }

    /**
     * Assign unit to class.
     */
    public function assignUnit(Request $request, SchoolClass $class)
    {
        $hod = Auth::user();

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to manage this class.');
        }

        $request->validate([
            'unit_id' => ['required', 'exists:units,id'],
        ]);

        // Check if unit belongs to the same department
        $unit = Unit::findOrFail($request->unit_id);
        if ($unit->department_id != $hod->department_id) {
            return redirect()->back()
                ->with('error', 'Unit must belong to your department.');
        }

        // Check if already assigned
        if ($class->units()->where('unit_id', $request->unit_id)->exists()) {
            return redirect()->back()
                ->with('error', 'This unit is already assigned to this class.');
        }

        $class->units()->attach($request->unit_id);

        return redirect()->back()
            ->with('success', 'Unit assigned to class successfully.');
    }

    /**
     * Remove unit from class.
     */
    public function removeUnit(Request $request, SchoolClass $class)
    {
        $hod = Auth::user();

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to manage this class.');
        }

        $request->validate([
            'unit_id' => ['required', 'exists:units,id'],
        ]);

        // Check if there are submissions for this unit in this class
        $hasSubmissions = \App\Models\PoeSubmission::where('unit_id', $request->unit_id)
            ->where('class_id', $class->id)
            ->exists();

        if ($hasSubmissions) {
            return redirect()->back()
                ->with('error', 'Cannot remove unit from class with associated submissions.');
        }

        $class->units()->detach($request->unit_id);

        return redirect()->back()
            ->with('success', 'Unit removed from class successfully.');
    }
}
