<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\SchoolClass;
use App\Models\User;
use App\Models\Department;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('hod.dashboard', [
                'activeTerm' => null,
                'department' => $hod->department,
                'stats' => [],
                'pendingApprovals' => collect(),
                'recentSubmissions' => collect(),
                'departmentProgress' => collect(),
            ])->with('warning', 'No active term found. Please contact your administrator.');
        }

        // Get HOD's department
        $department = $hod->department;
        
        if (!$department) {
            return view('hod.dashboard', [
                'activeTerm' => $activeTerm,
                'department' => null,
                'stats' => [],
                'pendingApprovals' => collect(),
                'recentSubmissions' => collect(),
                'departmentProgress' => collect(),
            ])->with('warning', 'You are not assigned to a department. Please contact your administrator.');
        }

        // Get department classes for active term - cache this
        $departmentClassIds = cache()->remember('hod.department.classes.' . $hod->id . '.' . $activeTerm->id, 600, function() use ($activeTerm, $department) {
            return SchoolClass::where('term_id', $activeTerm->id)
                ->where('department_id', $department->id)
                ->pluck('id')
                ->toArray();
        });

        if (empty($departmentClassIds)) {
            return view('hod.dashboard', [
                'activeTerm' => $activeTerm,
                'department' => $department,
                'stats' => [
                    'total_submissions' => 0,
                    'pending_approvals' => 0,
                    'approved' => 0,
                    'rejected' => 0,
                    'under_review' => 0,
                ],
                'pendingApprovals' => collect(),
                'recentSubmissions' => collect(),
                'departmentProgress' => collect(),
                'departmentStats' => [
                    'total_classes' => 0,
                    'total_students' => 0,
                    'total_trainers' => 0,
                ],
            ])->with('info', 'No classes found for your department in the active term.');
        }

        // Calculate statistics with caching - optimized single query
        $stats = cache()->remember('hod.dashboard.stats.' . $hod->id . '.' . $activeTerm->id, 300, function() use ($departmentClassIds) {
            // Use a single query with conditional aggregation instead of multiple queries
            $statusCounts = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->selectRaw('status, count(*) as count')
                ->groupBy('status')
                ->pluck('count', 'status')
                ->toArray();
            
            return [
                'total_submissions' => array_sum($statusCounts),
                'pending_approvals' => $statusCounts['trainer_reviewed'] ?? 0,
                'approved' => $statusCounts['hod_approved'] ?? 0,
                'rejected' => $statusCounts['rejected'] ?? 0,
                'under_review' => ($statusCounts['submitted'] ?? 0) + ($statusCounts['under_review'] ?? 0) + ($statusCounts['trainer_reviewed'] ?? 0),
                'exams_verified' => $statusCounts['exams_verified'] ?? 0,
                'validator_approved' => $statusCounts['validator_approved'] ?? 0,
            ];
        });

        // Pending approvals - cached and optimized
        $pendingApprovals = cache()->remember('hod.dashboard.pending_approvals.' . $hod->id . '.' . $activeTerm->id, 180, function() use ($departmentClassIds) {
            // Get submissions pending approval
            $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
                ->select('poe_submissions.id', 'poe_submissions.student_id', 'poe_submissions.unit_id', 'poe_submissions.class_id', 'poe_submissions.status', 'poe_submissions.submitted_at', 'poe_submissions.created_at')
                ->whereIn('class_id', $departmentClassIds)
                ->where('status', 'trainer_reviewed')
                ->with([
                    'student:id,name,email',
                    'unit:id,name,code',
                    'schoolClass:id,name',
                ])
                ->latest('submitted_at')
                ->limit(10)
                ->get();

            return $submissions->map(function($submission) {
                // Calculate days since submission
                if ($submission->submitted_at) {
                    $daysSinceSubmission = now()->diffInDays($submission->submitted_at);
                    $submission->days_since_submission = $daysSinceSubmission;
                    $submission->is_overdue = $daysSinceSubmission > 5;
                } else {
                    $submission->days_since_submission = null;
                    $submission->is_overdue = false;
                }
                return $submission;
            });
        });

        // Recent submissions - cached
        $recentSubmissions = cache()->remember('hod.dashboard.recent_submissions.' . $hod->id . '.' . $activeTerm->id, 180, function() use ($departmentClassIds) {
            return PoeSubmission::withoutGlobalScope('activeTerm')
                ->select('poe_submissions.id', 'poe_submissions.student_id', 'poe_submissions.unit_id', 'poe_submissions.class_id', 'poe_submissions.status', 'poe_submissions.submitted_at', 'poe_submissions.created_at')
                ->whereIn('class_id', $departmentClassIds)
                ->with(['student:id,name,email', 'unit:id,name,code', 'schoolClass:id,name'])
                ->latest('created_at')
                ->limit(10)
                ->get();
        });

        // Department progress by class - optimized with batch query
        $departmentProgress = cache()->remember('hod.dashboard.progress.' . $hod->id . '.' . $activeTerm->id, 300, function() use ($departmentClassIds) {
            $classes = SchoolClass::whereIn('id', $departmentClassIds)
                ->select('id', 'name', 'code')
                ->get();

            // Get all submission counts in a single query
            $submissionCounts = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->selectRaw('class_id, status, count(*) as count')
                ->groupBy('class_id', 'status')
                ->get()
                ->groupBy('class_id')
                ->map(function($group) {
                    return $group->pluck('count', 'status')->toArray();
                });

            return $classes->map(function($class) use ($submissionCounts) {
                $counts = $submissionCounts->get($class->id, []);
                $total = array_sum($counts);
                $approved = $counts['hod_approved'] ?? 0;
                $pending = $counts['trainer_reviewed'] ?? 0;
                
                return [
                    'class' => $class,
                    'total_submissions' => $total,
                    'approved' => $approved,
                    'pending_approval' => $pending,
                    'completion_rate' => $total > 0 ? round(($approved / $total) * 100, 1) : 0,
                ];
            })->sortByDesc('total_submissions')->values();
        });

        // Get department statistics - optimized and cached with actionable insights
        $departmentStats = cache()->remember('hod.dashboard.department_stats.' . $hod->id . '.' . $activeTerm->id, 600, function() use ($departmentClassIds, $department, $activeTerm) {
            // Get student count using subquery for better PostgreSQL compatibility
            $studentCount = DB::table('users')
                ->whereIn('id', function($query) use ($departmentClassIds) {
                    $query->select('student_id')
                        ->from('enrollments')
                        ->whereIn('class_id', $departmentClassIds)
                        ->distinct();
                })
                ->where('role', 'student')
                ->count();

            $totalClasses = count($departmentClassIds);
            $totalTrainers = User::where('role', 'trainer')
                ->where('department_id', $department->id)
                ->where('status', 'active')
                ->count();

            // Get classes with trainers assigned
            $classesWithTrainers = DB::table('trainer_unit_class')
                ->whereIn('class_id', $departmentClassIds)
                ->distinct('class_id')
                ->count('class_id');

            // Get units in department
            $totalUnits = \App\Models\Unit::where('department_id', $department->id)->count();
            
            // Get units with allocations
            $unitsWithAllocations = DB::table('trainer_unit_class')
                ->join('units', 'trainer_unit_class.unit_id', '=', 'units.id')
                ->where('units.department_id', $department->id)
                ->whereIn('trainer_unit_class.class_id', $departmentClassIds)
                ->distinct('units.id')
                ->count('units.id');

            // Get submission completion rate
            $totalSubmissions = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->count();
            
            $completedSubmissions = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->whereIn('status', ['hod_approved', 'exams_verified', 'validator_approved'])
                ->count();

            $completionRate = $totalSubmissions > 0 ? round(($completedSubmissions / $totalSubmissions) * 100, 1) : 0;

            return [
                'total_classes' => $totalClasses,
                'classes_with_trainers' => $classesWithTrainers,
                'classes_missing_trainers' => $totalClasses - $classesWithTrainers,
                'total_students' => $studentCount,
                'total_trainers' => $totalTrainers,
                'total_units' => $totalUnits,
                'units_with_allocations' => $unitsWithAllocations,
                'units_awaiting_allocation' => $totalUnits - $unitsWithAllocations,
                'submission_completion_rate' => $completionRate,
            ];
        });

        // Calculate department health
        $departmentHealth = cache()->remember('hod.dashboard.health.' . $hod->id . '.' . $activeTerm->id, 300, function() use ($stats, $departmentStats) {
            $issues = [];
            $status = 'good';
            
            if (($stats['pending_approvals'] ?? 0) > 0) {
                $issues[] = ($stats['pending_approvals'] ?? 0) . ' pending approval' . (($stats['pending_approvals'] ?? 0) > 1 ? 's' : '');
                $status = 'attention';
            }
            
            if (($departmentStats['units_awaiting_allocation'] ?? 0) > 0) {
                $issues[] = ($departmentStats['units_awaiting_allocation'] ?? 0) . ' unit' . (($departmentStats['units_awaiting_allocation'] ?? 0) > 1 ? 's' : '') . ' awaiting allocation';
                $status = 'attention';
            }
            
            if (($departmentStats['classes_missing_trainers'] ?? 0) > 0) {
                $issues[] = ($departmentStats['classes_missing_trainers'] ?? 0) . ' class' . (($departmentStats['classes_missing_trainers'] ?? 0) > 1 ? 'es' : '') . ' missing trainer';
                $status = 'attention';
            }

            // Check for critical issues
            if (($stats['pending_approvals'] ?? 0) > 10 || ($departmentStats['units_awaiting_allocation'] ?? $totalUnits) > ($departmentStats['total_units'] ?? 1) * 0.5) {
                $status = 'critical';
            }

            return [
                'status' => $status,
                'issues' => $issues,
                'message' => $status === 'good' ? 'All systems operational' : ($status === 'attention' ? 'Attention needed' : 'Critical issues detected'),
            ];
        });

        // Get recent activity - cached
        $recentActivity = cache()->remember('hod.dashboard.recent_activity.' . $hod->id . '.' . $activeTerm->id, 180, function() use ($departmentClassIds, $activeTerm) {
            $activities = [];
            
            // Recent submissions
            $recentSubs = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->with(['student:id,name', 'unit:id,name'])
                ->latest('created_at')
                ->limit(5)
                ->get();
            
            foreach ($recentSubs as $sub) {
                $activities[] = [
                    'type' => 'submission',
                    'message' => $sub->student->name . ' submitted ' . $sub->unit->name,
                    'time' => $sub->created_at,
                    'icon' => 'document',
                ];
            }

            // Recent status changes (submissions moved to trainer_reviewed or hod_approved)
            $recentStatusChanges = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->whereIn('status', ['trainer_reviewed', 'hod_approved'])
                ->with(['student:id,name', 'unit:id,name'])
                ->latest('updated_at')
                ->limit(5)
                ->get();

            foreach ($recentStatusChanges as $submission) {
                $statusLabel = $submission->status === 'trainer_reviewed' ? 'marked' : 'approved';
                $activities[] = [
                    'type' => 'status_change',
                    'message' => $submission->student->name . '\'s ' . $submission->unit->name . ' was ' . $statusLabel,
                    'time' => $submission->updated_at,
                    'icon' => 'check',
                ];
            }

            // Recent allocations
            $recentAllocations = DB::table('trainer_unit_class')
                ->join('users', 'trainer_unit_class.trainer_id', '=', 'users.id')
                ->join('units', 'trainer_unit_class.unit_id', '=', 'units.id')
                ->join('classes', 'trainer_unit_class.class_id', '=', 'classes.id')
                ->whereIn('classes.id', $departmentClassIds)
                ->select('users.name as trainer_name', 'units.name as unit_name', 'classes.name as class_name', 'trainer_unit_class.created_at')
                ->orderBy('trainer_unit_class.created_at', 'desc')
                ->limit(5)
                ->get();

            foreach ($recentAllocations as $alloc) {
                $activities[] = [
                    'type' => 'allocation',
                    'message' => $alloc->unit_name . ' allocated to ' . $alloc->class_name . ' (Trainer: ' . $alloc->trainer_name . ')',
                    'time' => $alloc->created_at,
                    'icon' => 'link',
                ];
            }

            // Sort by time and limit
            usort($activities, function($a, $b) {
                return strtotime($b['time']) - strtotime($a['time']);
            });

            return array_slice($activities, 0, 10);
        });

        // Get marks pending verification (trainer_reviewed status)
        $marksPendingVerification = cache()->remember('hod.dashboard.marks_pending.' . $hod->id . '.' . $activeTerm->id, 180, function() use ($departmentClassIds, $activeTerm) {
            return PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->where('status', 'trainer_reviewed')
                ->count();
        });

        return view('hod.dashboard', compact(
            'activeTerm',
            'department',
            'stats',
            'pendingApprovals',
            'recentSubmissions',
            'departmentProgress',
            'departmentStats',
            'departmentHealth',
            'recentActivity',
            'marksPendingVerification'
        ));
    }
}
