<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class ProfileController extends Controller
{
    /**
     * Display the HOD's profile.
     */
    public function show()
    {
        $user = Auth::user();
        $user->load('department:id,name,code');
        
        // Get department statistics
        $departmentStats = [
            'total_classes' => \App\Models\SchoolClass::where('department_id', $user->department_id)
                ->where('term_id', \App\Services\ActiveTermService::getActiveTermId())
                ->count(),
            'total_students' => \App\Models\User::where('role', 'student')
                ->whereHas('enrollments', function($query) use ($user) {
                    $query->whereHas('schoolClass', function($q) use ($user) {
                        $q->where('department_id', $user->department_id)
                          ->where('term_id', \App\Services\ActiveTermService::getActiveTermId());
                    });
                })
                ->distinct()
                ->count(),
            'total_trainers' => \App\Models\User::where('role', 'trainer')
                ->where('department_id', $user->department_id)
                ->where('status', 'active')
                ->count(),
        ];
        
        return view('hod.profile.show', compact('user', 'departmentStats'));
    }

    /**
     * Show the form for editing the profile.
     */
    public function edit()
    {
        $user = Auth::user();
        $user->load('department:id,name,code');
        
        return view('hod.profile.edit', compact('user'));
    }

    /**
     * Update the HOD's profile.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20'],
            'current_password' => ['nullable', 'required_with:password', 'current-password'],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
        ]);

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone = $request->phone;

        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }

        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
        }

        $user->save();

        return redirect()->route('hod.profile.show')
            ->with('success', 'Profile updated successfully.');
    }
}
