<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Department;
use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class UserController extends Controller
{
    /**
     * Display a listing of users in HOD's department.
     */
    public function index(Request $request)
    {
        $hod = auth()->user();
        
        if (!$hod->department_id) {
            return redirect()->route('hod.dashboard')
                ->with('warning', 'You are not assigned to a department.');
        }

        $query = User::where('department_id', $hod->department_id);

        // Filter by role if provided (HOD can manage: trainer, student, validator)
        if ($request->has('role') && $request->role && $request->role !== 'all') {
            $query->where('role', $request->role);
        }

        // Filter by status if provided
        if ($request->has('status') && $request->status && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // Search by name, email, or admission number
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('admission_number', 'like', "%{$search}%");
            });
        }

        $users = $query->with('department:id,name')->latest()->paginate(15);
        
        $selectedRole = $request->role ?? 'all';
        $selectedStatus = $request->status ?? 'all';
        $search = $request->search ?? '';

        // Get counts by role for department only
        $roleCounts = cache()->remember('hod.users.role_counts.' . $hod->id, 300, function () use ($hod) {
            return User::where('department_id', $hod->department_id)
                ->selectRaw('role, count(*) as count')
                ->groupBy('role')
                ->get()
                ->pluck('count', 'role')
                ->toArray();
        });
        
        // Ensure all relevant roles are present
        $allRoles = ['trainer', 'student', 'validator'];
        foreach ($allRoles as $role) {
            if (!isset($roleCounts[$role])) {
                $roleCounts[$role] = 0;
            }
        }

        return view('hod.users.index', compact(
            'users', 
            'selectedRole', 
            'selectedStatus',
            'search',
            'roleCounts'
        ));
    }

    /**
     * Show the form for creating a new user in HOD's department.
     */
    public function create()
    {
        $hod = auth()->user();
        
        if (!$hod->department_id) {
            return redirect()->route('hod.users.index')
                ->with('warning', 'You are not assigned to a department.');
        }

        // Only show units from HOD's department
        $units = Unit::where('department_id', $hod->department_id)
            ->select('id', 'name', 'code', 'department_id')
            ->orderBy('name')
            ->get();
            
        return view('hod.users.create', compact('units'));
    }

    /**
     * Store a newly created user in storage.
     */
    public function store(Request $request)
    {
        $hod = auth()->user();
        
        if (!$hod->department_id) {
            return redirect()->back()
                ->with('error', 'You are not assigned to a department.');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'in:trainer,student,validator'],
            'phone' => ['nullable', 'string', 'max:20'],
            'status' => ['required', 'in:active,inactive,suspended'],
            'admission_number' => ['nullable', 'string', 'max:50', 'unique:users,admission_number'],
            'unit_ids' => ['nullable', 'array'],
            'unit_ids.*' => ['exists:units,id'],
        ]);

        // Validate units belong to HOD's department
        if ($request->has('unit_ids') && !empty($request->unit_ids)) {
            $validUnitIds = Unit::where('department_id', $hod->department_id)
                ->whereIn('id', $request->unit_ids)
                ->pluck('id')
                ->toArray();
            
            if (count($validUnitIds) !== count($request->unit_ids)) {
                return redirect()->back()
                    ->with('error', 'One or more selected units do not belong to your department.')
                    ->withInput();
            }
        }

        $user = User::create([
            'name' => $request->name,
            'first_name' => null,
            'last_name' => null,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'role' => $request->role,
            'department_id' => $hod->department_id, // Always set to HOD's department
            'phone' => $request->phone,
            'status' => $request->status,
            'admission_number' => $request->admission_number,
            'email_verified_at' => now(),
        ]);

        // If trainer, assign units if provided
        if ($user->isTrainer() && $request->has('unit_ids') && !empty($request->unit_ids)) {
            $user->units()->attach($request->unit_ids);
        }

        // Clear cache
        cache()->forget('hod.users.role_counts.' . $hod->id);
        cache()->forget('hod.dashboard.department_stats.' . $hod->id);

        return redirect()->route('hod.users.index', ['role' => $request->role])
            ->with('success', 'User created successfully.');
    }

    /**
     * Display the specified user.
     */
    public function show(User $user)
    {
        $hod = auth()->user();
        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();
        
        // Verify user belongs to HOD's department
        if ($user->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this user.');
        }

        $user->load([
            'department:id,name',
            'enrollments' => function($query) use ($activeTerm) {
                $query->select('id', 'student_id', 'class_id', 'status', 'enrolled_at')
                      ->with(['schoolClass:id,name,term_id,department_id,level_id' => function($q) use ($activeTerm) {
                          $q->with(['term:id,name', 'department:id,name', 'level:id,name']);
                          if ($activeTerm) {
                              $q->where('term_id', $activeTerm->id);
                          }
                      }]);
            },
            'poeSubmissions' => function($query) use ($activeTerm) {
                $query->select('id', 'student_id', 'unit_id', 'class_id', 'status', 'version', 'submitted_at')
                      ->with(['unit:id,name', 'schoolClass:id,name']);
                if ($activeTerm) {
                    $query->whereHas('schoolClass', function($q) use ($activeTerm) {
                        $q->where('term_id', $activeTerm->id);
                    });
                }
            }
        ]);

        // Get assigned units for trainers - filtered by active term
        $assignedUnits = collect();
        if ($user->isTrainer() && $activeTerm) {
            // Get classes in active term for HOD's department
            $departmentClassIds = \App\Models\SchoolClass::where('term_id', $activeTerm->id)
                ->where('department_id', $hod->department_id)
                ->pluck('id')
                ->toArray();

            // Get unit IDs assigned to this trainer in active term classes
            $unitIds = \Illuminate\Support\Facades\DB::table('trainer_unit_class')
                ->where('trainer_id', $user->id)
                ->whereIn('class_id', $departmentClassIds)
                ->distinct()
                ->pluck('unit_id')
                ->toArray();

            // Load units
            if (!empty($unitIds)) {
                $assignedUnits = \App\Models\Unit::whereIn('id', $unitIds)
                    ->select('id', 'name', 'code')
                    ->orderBy('name')
                    ->get();
            }
        }
        
        return view('hod.users.show', compact('user', 'assignedUnits', 'activeTerm'));
    }

    /**
     * Show the form for editing the specified user.
     */
    public function edit(User $user)
    {
        $hod = auth()->user();
        
        // Verify user belongs to HOD's department
        if ($user->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to edit this user.');
        }

        // Only show units from HOD's department
        $units = Unit::where('department_id', $hod->department_id)
            ->select('id', 'name', 'code', 'department_id')
            ->orderBy('name')
            ->get();
            
        $user->load('units:id');
        $assignedUnitIds = $user->units->pluck('id')->toArray();
        
        return view('hod.users.edit', compact('user', 'units', 'assignedUnitIds'));
    }

    /**
     * Update the specified user in storage.
     */
    public function update(Request $request, User $user)
    {
        $hod = auth()->user();
        
        // Verify user belongs to HOD's department
        if ($user->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to update this user.');
        }

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
            'role' => ['required', 'in:trainer,student,validator'],
            'phone' => ['nullable', 'string', 'max:20'],
            'status' => ['required', 'in:active,inactive,suspended'],
            'admission_number' => ['nullable', 'string', 'max:50', 'unique:users,admission_number,' . $user->id],
            'unit_ids' => ['nullable', 'array'],
            'unit_ids.*' => ['exists:units,id'],
        ]);

        // Validate units belong to HOD's department
        if ($request->has('unit_ids') && !empty($request->unit_ids)) {
            $validUnitIds = Unit::where('department_id', $hod->department_id)
                ->whereIn('id', $request->unit_ids)
                ->pluck('id')
                ->toArray();
            
            if (count($validUnitIds) !== count($request->unit_ids)) {
                return redirect()->back()
                    ->with('error', 'One or more selected units do not belong to your department.')
                    ->withInput();
            }
        }

        $wasTrainer = $user->isTrainer();

        $data = [
            'name' => $request->name,
            'first_name' => null,
            'last_name' => null,
            'email' => $request->email,
            'role' => $request->role,
            'department_id' => $hod->department_id, // Always keep in HOD's department
            'phone' => $request->phone,
            'status' => $request->status,
            'admission_number' => $request->admission_number,
        ];

        if ($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }

        $user->update($data);

        // Handle unit assignments for trainers
        if ($user->isTrainer()) {
            if ($request->has('unit_ids')) {
                $user->units()->sync($request->unit_ids);
            } else {
                $user->units()->sync([]);
            }
        } else {
            // If user is no longer a trainer, remove all unit assignments
            if ($wasTrainer && !$user->isTrainer()) {
                $user->units()->detach();
            }
        }

        // Clear cache
        cache()->forget('hod.users.role_counts.' . $hod->id);
        cache()->forget('hod.dashboard.department_stats.' . $hod->id);

        return redirect()->route('hod.users.index', ['role' => $request->role])
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified user from storage.
     */
    public function destroy(User $user)
    {
        $hod = auth()->user();
        
        // Verify user belongs to HOD's department
        if ($user->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to delete this user.');
        }

        if ($user->id === auth()->id()) {
            return redirect()->route('hod.users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        $user->delete();

        // Clear cache
        cache()->forget('hod.users.role_counts.' . $hod->id);
        cache()->forget('hod.dashboard.department_stats.' . $hod->id);

        return redirect()->route('hod.users.index')
            ->with('success', 'User deleted successfully.');
    }
}
