<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\Enrollment;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        $student = Auth::user();
        $activeTermId = ActiveTermService::getActiveTermId();

        // Get student's enrollments for active term - optimized with select
        $enrollments = Enrollment::select('id', 'student_id', 'class_id', 'enrolled_at', 'status')
            ->where('student_id', $student->id)
            ->whereHas('schoolClass', function($query) use ($activeTermId) {
                if ($activeTermId) {
                    $query->where('term_id', $activeTermId);
                }
            })
            ->with([
                'schoolClass:id,name,code,department_id,level_id' => [
                    'department:id,name',
                    'level:id,name',
                    'units:id,name,code'
                ]
            ])
            ->get();

        // Get all units from enrolled classes - optimized batch query
        $classIds = $enrollments->pluck('class_id');
        $unitIds = \Illuminate\Support\Facades\DB::table('class_unit')
            ->whereIn('class_id', $classIds)
            ->distinct()
            ->pluck('unit_id');
        
        // Batch load assignments with type breakdown for all units - ONLY for student's enrolled classes
        $assignmentsData = Assignment::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
            ->where('is_published', true)
            ->where(function($query) {
                $query->whereNull('start_date')
                      ->orWhere('start_date', '<=', now());
            })
            ->selectRaw('unit_id, type, count(*) as count')
            ->groupBy('unit_id', 'type')
            ->get();
        
        $assignmentsCounts = [];
        $writtenCounts = [];
        $practicalCounts = [];
        
        foreach ($assignmentsData as $data) {
            $unitId = $data->unit_id;
            $assignmentsCounts[$unitId] = ($assignmentsCounts[$unitId] ?? 0) + $data->count;
            
            if ($data->type === 'practical') {
                $practicalCounts[$unitId] = ($practicalCounts[$unitId] ?? 0) + $data->count;
            } else {
                $writtenCounts[$unitId] = ($writtenCounts[$unitId] ?? 0) + $data->count;
            }
        }
        
        // Get units with their class associations
        $unitClassMap = \Illuminate\Support\Facades\DB::table('class_unit')
            ->whereIn('class_id', $classIds)
            ->whereIn('unit_id', $unitIds)
            ->join('classes', 'class_unit.class_id', '=', 'classes.id')
            ->select('class_unit.unit_id', 'classes.name as class_name')
            ->get()
            ->groupBy('unit_id');
        
        $units = \App\Models\Unit::whereIn('id', $unitIds)
            ->with('department:id,name,code')
            ->get()
            ->map(function($unit) use ($assignmentsCounts, $writtenCounts, $practicalCounts, $unitClassMap) {
                $unit->assignments_count = $assignmentsCounts[$unit->id] ?? 0;
                $unit->written_count = $writtenCounts[$unit->id] ?? 0;
                $unit->practical_count = $practicalCounts[$unit->id] ?? 0;
                $unit->class_name = $unitClassMap[$unit->id]->first()->class_name ?? '';
                return $unit;
            })
            ->sortBy('name')
            ->values();
        
        $unitIds = $units->pluck('id');

        // Get submission statistics
        $stats = cache()->remember('student.dashboard.stats.' . $student->id, 300, function() use ($student, $activeTermId) {
            $baseQuery = PoeSubmission::where('student_id', $student->id);
            
            if ($activeTermId) {
                $baseQuery->whereHas('schoolClass', function($q) use ($activeTermId) {
                    $q->where('term_id', $activeTermId);
                });
            }
            
            return [
                'total_submissions' => (clone $baseQuery)->count(),
                'draft' => (clone $baseQuery)->where('status', 'draft')->count(),
                'submitted' => (clone $baseQuery)->where('status', 'submitted')->count(),
                'under_review' => (clone $baseQuery)->where('status', 'under_review')->count(),
                'trainer_reviewed' => (clone $baseQuery)->where('status', 'trainer_reviewed')->count(),
                'hod_approved' => (clone $baseQuery)->where('status', 'hod_approved')->count(),
                'exams_verified' => (clone $baseQuery)->where('status', 'exams_verified')->count(),
                'validator_approved' => (clone $baseQuery)->where('status', 'validator_approved')->count(),
                'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
            ];
        });

        // Recent submissions - cached
        $recentSubmissions = cache()->remember('student.dashboard.recent_submissions.' . $student->id, 180, function() use ($student, $activeTermId) {
            return PoeSubmission::select('id', 'unit_id', 'class_id', 'status', 'submitted_at', 'created_at')
                ->where('student_id', $student->id)
                ->when($activeTermId, function($query) use ($activeTermId) {
                    $query->whereHas('schoolClass', function($q) use ($activeTermId) {
                        $q->where('term_id', $activeTermId);
                    });
                })
                ->with([
                    'unit:id,name,code',
                    'schoolClass:id,name'
                ])
                ->latest()
                ->limit(5)
                ->get();
        });

        // Get units that need submission (units in enrolled classes without submissions)
        $submittedUnitIds = PoeSubmission::where('student_id', $student->id)
            ->when($activeTermId, function($query) use ($activeTermId) {
                $query->whereHas('schoolClass', function($q) use ($activeTermId) {
                    $q->where('term_id', $activeTermId);
                });
            })
            ->pluck('unit_id')
            ->unique();

        $pendingUnits = collect();
        foreach ($enrollments as $enrollment) {
            foreach ($enrollment->schoolClass->units as $unit) {
                if (!$submittedUnitIds->contains($unit->id)) {
                    $pendingUnits->push([
                        'unit' => $unit,
                        'class' => $enrollment->schoolClass,
                    ]);
                }
            }
        }

        // Get practical assessment statistics
        $practicalStats = cache()->remember('student.dashboard.practical_stats.' . $student->id, 300, function() use ($student, $unitIds, $classIds) {
            $practicalAssignments = Assignment::whereIn('unit_id', $unitIds)
                ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
                ->where('type', 'practical')
                ->where('is_published', true)
                ->pluck('id');
            
            $submissions = AssignmentSubmission::where('student_id', $student->id)
                ->whereIn('assignment_id', $practicalAssignments)
                ->get();
            
            return [
                'marked' => $submissions->where('marking_status', 'marked')->count(),
                'pending' => $submissions->whereIn('marking_status', ['not_marked', null])->count() + 
                            ($practicalAssignments->count() - $submissions->count()),
                'absent' => $submissions->where('marking_status', 'absent')->count(),
                'total' => $practicalAssignments->count(),
            ];
        });
        
        // Get assessment mode breakdown (written vs practical)
        $assessmentMode = cache()->remember('student.dashboard.assessment_mode.' . $student->id, 300, function() use ($unitIds, $classIds) {
            $allAssignments = Assignment::whereIn('unit_id', $unitIds)
                ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
                ->where('is_published', true)
                ->where(function($query) {
                    $query->whereNull('start_date')
                          ->orWhere('start_date', '<=', now());
                })
                ->get();
            
            return [
                'has_written' => $allAssignments->where('type', '!=', 'practical')->count() > 0,
                'has_practical' => $allAssignments->where('type', 'practical')->count() > 0,
                'written_count' => $allAssignments->where('type', '!=', 'practical')->count(),
                'practical_count' => $allAssignments->where('type', 'practical')->count(),
            ];
        });
        
        // Get pending assignments with due dates (for urgency indicators)
        $pendingAssignments = Assignment::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
            ->where('is_published', true)
            ->where('type', '!=', 'practical') // Only written assignments need submission
            ->where(function($query) {
                $query->whereNull('start_date')
                      ->orWhere('start_date', '<=', now());
            })
            ->whereDoesntHave('submissions', function($q) use ($student) {
                $q->where('student_id', $student->id)
                  ->where('status', 'submitted');
            })
            ->with('unit:id,name,code')
            ->get()
            ->map(function($assignment) {
                $assignment->days_until_due = $assignment->due_date 
                    ? now()->diffInDays($assignment->due_date, false) 
                    : null;
                $assignment->is_overdue = $assignment->due_date && $assignment->due_date->isPast();
                return $assignment;
            })
            ->sortBy(function($assignment) {
                if ($assignment->is_overdue) return -1;
                return $assignment->days_until_due ?? 999;
            })
            ->take(5);
        
        // Get progress breakdown
        $progressBreakdown = cache()->remember('student.dashboard.progress.' . $student->id, 300, function() use ($student, $unitIds, $classIds) {
            $allAssignments = Assignment::whereIn('unit_id', $unitIds)
                ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
                ->where('is_published', true)
                ->get();
            
            $writtenAssignments = $allAssignments->where('type', '!=', 'practical');
            $practicalAssignments = $allAssignments->where('type', 'practical');
            
            $writtenSubmitted = AssignmentSubmission::where('student_id', $student->id)
                ->whereIn('assignment_id', $writtenAssignments->pluck('id'))
                ->where('status', 'submitted')
                ->count();
            
            $practicalMarked = AssignmentSubmission::where('student_id', $student->id)
                ->whereIn('assignment_id', $practicalAssignments->pluck('id'))
                ->where('marking_status', 'marked')
                ->count();
            
            $writtenTotal = $writtenAssignments->count();
            $practicalTotal = $practicalAssignments->count();
            
            return [
                'written' => $writtenTotal > 0 ? round(($writtenSubmitted / $writtenTotal) * 100) : 0,
                'practical' => $practicalTotal > 0 ? round(($practicalMarked / $practicalTotal) * 100) : 0,
                'overall' => ($writtenTotal + $practicalTotal) > 0 
                    ? round((($writtenSubmitted + $practicalMarked) / ($writtenTotal + $practicalTotal)) * 100) 
                    : 0,
            ];
        });
        
        // Recent activity (submissions + practical marks + feedback)
        $recentActivity = collect();
        
        // Add recent submissions
        foreach ($recentSubmissions as $submission) {
            if ($submission->unit) {
                $recentActivity->push([
                    'type' => 'submission',
                    'title' => 'Submitted assignment',
                    'unit' => $submission->unit->name,
                    'date' => $submission->submitted_at ?? $submission->created_at,
                    'status' => $submission->status,
                ]);
            }
        }
        
        // Add recent practical marks
        if ($unitIds->count() > 0) {
            $recentPracticalMarks = AssignmentSubmission::where('student_id', $student->id)
                ->whereHas('assignment', function($q) use ($unitIds, $classIds) {
                    $q->whereIn('unit_id', $unitIds)
                      ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for student's enrolled classes
                      ->where('type', 'practical');
                })
                ->where('marking_status', 'marked')
                ->whereNotNull('graded_at')
                ->with(['assignment.unit:id,name,code'])
                ->latest('graded_at')
                ->limit(3)
                ->get();
            
            foreach ($recentPracticalMarks as $mark) {
                if ($mark->assignment && $mark->assignment->unit) {
                    $recentActivity->push([
                        'type' => 'practical_marked',
                        'title' => 'Practical marked',
                        'unit' => $mark->assignment->unit->name ?? 'Unknown',
                        'date' => $mark->graded_at,
                        'marks' => $mark->marks,
                    ]);
                }
            }
        }
        
        $recentActivity = $recentActivity->sortByDesc(function($item) {
            return $item['date'] ? $item['date']->timestamp : 0;
        })->take(5)->values();
        
        $activeTerm = ActiveTermService::getActiveTerm();

        return view('student.dashboard', compact(
            'stats',
            'enrollments',
            'recentSubmissions',
            'recentActivity',
            'pendingUnits',
            'pendingAssignments',
            'activeTerm',
            'units',
            'practicalStats',
            'assessmentMode',
            'progressBreakdown'
        ));
    }
}
