<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\Enrollment;
use App\Models\PoeSubmission;
use App\Models\Unit;
use App\Services\ActiveTermService;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use iio\libmergepdf\Merger;

class ProgressController extends Controller
{
    /**
     * Display the student's progress overview.
     */
    public function index()
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return view('student.progress.index', [
                'activeTerm' => null,
                'progress' => [],
                'stats' => [],
                'classProgress' => collect(),
            ])->with('warning', 'No active term found. Please contact your administrator.');
        }

        // Get student's enrollments for the active term
        $enrollments = $user->enrollments()
            ->whereHas('schoolClass', function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with([
                'schoolClass:id,name,code,department_id,level_id' => [
                    'department:id,name,code',
                    'level:id,name',
                    'units:id,name,code,department_id' => [
                        'department:id,name'
                    ]
                ]
            ])
            ->get();

        // Get all submissions for the active term
        $submissions = $user->poeSubmissions()
            ->whereHas('schoolClass', function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with([
                'unit:id,name,code',
                'schoolClass:id,name,code',
                'reviews:id,poe_submission_id,reviewer_id,marks,feedback,status' => [
                    'reviewer:id,name'
                ],
                'validation:id,poe_submission_id,validator_id,status,comments' => [
                    'validator:id,name'
                ]
            ])
            ->get();

        // Calculate overall statistics
        $totalUnits = $enrollments->sum(function($enrollment) {
            return $enrollment->schoolClass->units->count();
        });

        $submittedUnits = $submissions->count();
        $pendingUnits = max(0, $totalUnits - $submittedUnits);
        $completionPercentage = $totalUnits > 0 ? round(($submittedUnits / $totalUnits) * 100, 1) : 0;

        // Status breakdown
        $statusBreakdown = [
            'draft' => $submissions->where('status', 'draft')->count(),
            'submitted' => $submissions->where('status', 'submitted')->count(),
            'under_review' => $submissions->where('status', 'under_review')->count(),
            'trainer_reviewed' => $submissions->where('status', 'trainer_reviewed')->count(),
            'hod_approved' => $submissions->where('status', 'hod_approved')->count(),
            'exams_verified' => $submissions->where('status', 'exams_verified')->count(),
            'validator_approved' => $submissions->where('status', 'validator_approved')->count(),
            'rejected' => $submissions->where('status', 'rejected')->count(),
        ];

        $approvedCount = $statusBreakdown['validator_approved'];
        $approvedPercentage = $totalUnits > 0 ? round(($approvedCount / $totalUnits) * 100, 1) : 0;

        // Calculate marks (if any reviews have marks)
        $totalMarks = $submissions->sum(function($submission) {
            return $submission->reviews->sum('marks') ?? 0;
        });
        $submissionsWithMarks = $submissions->filter(function($submission) {
            return $submission->reviews->whereNotNull('marks')->count() > 0;
        })->count();
        $averageMarks = $submissionsWithMarks > 0 ? round($totalMarks / $submissionsWithMarks, 1) : 0;

        // Progress by class
        $classProgress = $enrollments->map(function($enrollment) use ($submissions) {
            $classUnits = $enrollment->schoolClass->units;
            $classUnitIds = $classUnits->pluck('id');
            $classSubmissions = $submissions->whereIn('unit_id', $classUnitIds)
                ->where('class_id', $enrollment->class_id);

            $total = $classUnits->count();
            $submitted = $classSubmissions->count();
            $approved = $classSubmissions->where('status', 'validator_approved')->count();
            $completion = $total > 0 ? round(($submitted / $total) * 100, 1) : 0;

            return [
                'class' => $enrollment->schoolClass,
                'enrollment' => $enrollment,
                'total_units' => $total,
                'submitted_units' => $submitted,
                'approved_units' => $approved,
                'pending_units' => $total - $submitted,
                'completion_percentage' => $completion,
                'submissions' => $classSubmissions,
            ];
        });

        // Recent activity (submissions ordered by date)
        $recentActivity = $submissions
            ->sortByDesc(function($submission) {
                return $submission->submitted_at ?? $submission->created_at;
            })
            ->take(10)
            ->values();

        // Overall stats
        $stats = [
            'total_units' => $totalUnits,
            'submitted_units' => $submittedUnits,
            'pending_units' => $pendingUnits,
            'approved_units' => $approvedCount,
            'completion_percentage' => $completionPercentage,
            'approved_percentage' => $approvedPercentage,
            'total_submissions' => $submissions->count(),
            'average_marks' => $averageMarks,
            'submissions_with_marks' => $submissionsWithMarks,
        ];

        // Progress data for charts/visualization
        $progress = [
            'status_breakdown' => $statusBreakdown,
            'completion_data' => [
                'completed' => $submittedUnits,
                'pending' => $pendingUnits,
            ],
        ];

        // Units in the student's classes (for Download Unit Documents dropdown)
        $progressUnits = $classProgress->flatMap(function ($item) {
            return $item['class']->units ?? collect();
        })->unique('id')->sortBy('name')->values();

        // Per-unit counts for "Practicals (5)", "Exams (2)", "Portfolio (3)" when unit is selected
        $unitDocumentCounts = $progressUnits->count() > 0
            ? $this->buildUnitDocumentCounts($user, $progressUnits, $enrollments, $activeTerm)
            : [];

        return view('student.progress.index', compact(
            'activeTerm',
            'stats',
            'progress',
            'classProgress',
            'recentActivity',
            'progressUnits',
            'unitDocumentCounts'
        ));
    }

    /**
     * Unified download: Step 1 select unit, Step 2 select type (practicals|exams|portfolio), Step 3 download PDF.
     */
    public function unitDocumentPdf(\Illuminate\Http\Request $request)
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return redirect()->route('student.progress.index')
                ->with('warning', 'No active term found. Cannot generate PDF.');
        }

        $request->validate([
            'unit' => 'required|exists:units,id',
            'type' => 'required|in:practicals,exams,portfolio',
        ]);

        $enrollments = $user->enrollments()
            ->whereHas('schoolClass', function ($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with(['schoolClass:id,name,code' => ['units:id,name,code']])
            ->get();

        $allowedUnitIds = $enrollments->flatMap(function ($e) {
            return ($e->schoolClass->units ?? collect())->pluck('id');
        })->unique()->values()->all();

        if (!in_array((int) $request->unit, $allowedUnitIds, true)) {
            abort(403, 'You can only download documents for units in your class.');
        }

        $unit = Unit::findOrFail($request->unit);
        $classIds = $enrollments->pluck('class_id')->unique()->values()->all();
        $type = $request->type;

        if ($type === 'practicals') {
            return $this->buildPracticalsPdfResponse($user, $activeTerm, $unit, $classIds);
        }
        if ($type === 'exams') {
            return $this->buildExamsPdfResponse($user, $activeTerm, $unit, $classIds);
        }
        return $this->buildPortfolioPdfResponse($user, $activeTerm, $unit, $classIds);
    }

    /**
     * Build per-unit counts for practicals (completed assessor tools), exams (PDF attachments), portfolio (POE submissions).
     */
    protected function buildUnitDocumentCounts($user, $progressUnits, $enrollments, $activeTerm)
    {
        $classIds = $enrollments->pluck('class_id')->unique()->values()->all();
        $unitIds = $progressUnits->pluck('id')->all();

        $practicalsCount = [];
        $examCount = [];
        $portfolioCount = [];

        foreach ($unitIds as $uid) {
            $practicalsCount[$uid] = 0;
            $examCount[$uid] = 0;
            $portfolioCount[$uid] = 0;
        }

        $practicalAssignments = Assignment::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds)
            ->where('type', 'practical')
            ->where('is_published', true)
            ->get(['id', 'unit_id']);
        $submissionsWithTool = AssignmentSubmission::whereIn('assignment_id', $practicalAssignments->pluck('id'))
            ->where('student_id', $user->id)
            ->whereNotNull('completed_assessor_tool_path')
            ->get(['assignment_id', 'completed_assessor_tool_path']);
        foreach ($practicalAssignments as $a) {
            $sub = $submissionsWithTool->where('assignment_id', $a->id)->first();
            if ($sub && $sub->completed_assessor_tool_path && Str::endsWith(strtolower($sub->completed_assessor_tool_path), '.pdf') && Storage::disk('public')->exists($sub->completed_assessor_tool_path)) {
                $practicalsCount[$a->unit_id] = ($practicalsCount[$a->unit_id] ?? 0) + 1;
            }
        }

        $examAssignments = Assignment::whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds)
            ->where('type', '!=', 'practical')
            ->where('is_published', true)
            ->get(['id', 'unit_id']);
        $examSubmissions = AssignmentSubmission::whereIn('assignment_id', $examAssignments->pluck('id'))
            ->where('student_id', $user->id)
            ->get(['id', 'assignment_id', 'attachments']);
        foreach ($examAssignments as $a) {
            $sub = $examSubmissions->where('assignment_id', $a->id)->first();
            if ($sub && is_array($sub->attachments)) {
                $hasPdf = false;
                foreach ($sub->attachments as $path) {
                    if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                        $hasPdf = true;
                        break;
                    }
                }
                if ($hasPdf) {
                    $examCount[$a->unit_id] = ($examCount[$a->unit_id] ?? 0) + 1;
                }
            }
        }

        $poeCounts = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('student_id', $user->id)
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds)
            ->selectRaw('unit_id, count(*) as c')
            ->groupBy('unit_id')
            ->pluck('c', 'unit_id');
        foreach ($unitIds as $uid) {
            $portfolioCount[$uid] = $poeCounts->get($uid, 0);
        }

        $out = [];
        foreach ($unitIds as $uid) {
            $out[$uid] = [
                'practicals' => $practicalsCount[$uid] ?? 0,
                'exams' => $examCount[$uid] ?? 0,
                'portfolio' => $portfolioCount[$uid] ?? 0,
            ];
        }
        return $out;
    }

    protected function buildPracticalsPdfResponse($user, $activeTerm, $unit, $classIds)
    {
        $practicalAssignments = Assignment::where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->where('type', 'practical')
            ->where('is_published', true)
            ->orderBy('due_date')->orderBy('id')
            ->get();

        $pdfPaths = [];
        foreach ($practicalAssignments as $assignment) {
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $user->id)
                ->first();
            if (!$submission || !$submission->completed_assessor_tool_path) {
                continue;
            }
            $path = $submission->completed_assessor_tool_path;
            if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                $pdfPaths[] = $path;
            }
        }

        if (empty($pdfPaths)) {
            $pdf = Pdf::loadView('student.progress.units-checklist-empty', [
                'user' => $user,
                'activeTerm' => $activeTerm,
                'unit' => $unit,
            ]);
            $pdf->setPaper('a4', 'portrait');
            $filename = 'Assessor_tools_' . preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit') . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);
        }

        $tempDir = 'progress-pdf-temp/' . uniqid('', true);
        Storage::disk('local')->makeDirectory($tempDir);
        try {
            $merger = new Merger();
            $dividerPdf = Pdf::loadView('student.progress.units-checklist-divider', compact('unit'));
            $dividerPdf->setPaper('a4', 'portrait');
            $dividerPath = $tempDir . '/divider_' . $unit->id . '.pdf';
            Storage::disk('local')->put($dividerPath, $dividerPdf->output());
            $merger->addFile(Storage::disk('local')->path($dividerPath));
            foreach ($pdfPaths as $path) {
                $abs = Storage::disk('public')->path($path);
                if (is_readable($abs)) {
                    $merger->addFile($abs);
                }
            }
            $mergedContent = $merger->merge();
        } finally {
            Storage::disk('local')->deleteDirectory($tempDir);
        }
        if (!$mergedContent) {
            return redirect()->route('student.progress.index')->with('error', 'Could not generate PDF.');
        }
        $filename = 'Assessor_tools_' . preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit') . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
        return response($mergedContent, 200, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    protected function buildExamsPdfResponse($user, $activeTerm, $unit, $classIds)
    {
        $examAssignments = Assignment::where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->where('is_published', true)
            ->where('type', '!=', 'practical')
            ->orderBy('start_date')->orderBy('id')
            ->get();

        $pdfPaths = [];
        foreach ($examAssignments as $assignment) {
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $user->id)
                ->first();
            if (!$submission || !is_array($submission->attachments)) {
                continue;
            }
            foreach ($submission->attachments as $path) {
                if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                    $pdfPaths[] = $path;
                    break;
                }
            }
        }

        if (empty($pdfPaths)) {
            $pdf = Pdf::loadView('student.progress.units-exams-empty', [
                'user' => $user,
                'activeTerm' => $activeTerm,
                'unit' => $unit,
            ]);
            $pdf->setPaper('a4', 'portrait');
            $filename = 'Exams_' . preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit') . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);
        }

        $tempDir = 'progress-pdf-temp/' . uniqid('', true);
        Storage::disk('local')->makeDirectory($tempDir);
        try {
            $merger = new Merger();
            $dividerPdf = Pdf::loadView('student.progress.units-checklist-divider', compact('unit'));
            $dividerPdf->setPaper('a4', 'portrait');
            $dividerPath = $tempDir . '/divider_' . $unit->id . '.pdf';
            Storage::disk('local')->put($dividerPath, $dividerPdf->output());
            $merger->addFile(Storage::disk('local')->path($dividerPath));
            foreach ($pdfPaths as $path) {
                $abs = Storage::disk('public')->path($path);
                if (is_readable($abs)) {
                    $merger->addFile($abs);
                }
            }
            $mergedContent = $merger->merge();
        } finally {
            Storage::disk('local')->deleteDirectory($tempDir);
        }
        if (!$mergedContent) {
            return redirect()->route('student.progress.index')->with('error', 'Could not generate PDF.');
        }
        $filename = 'Exams_' . preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit') . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
        return response($mergedContent, 200, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    protected function buildPortfolioPdfResponse($user, $activeTerm, $unit, $classIds)
    {
        $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('student_id', $user->id)
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->with(['schoolClass:id,name,code'])
            ->orderBy('submitted_at')
            ->orderBy('id')
            ->get();

        $pdf = Pdf::loadView('student.progress.units-portfolio-pdf', [
            'user' => $user,
            'activeTerm' => $activeTerm,
            'unit' => $unit,
            'submissions' => $submissions,
        ]);
        $pdf->setPaper('a4', 'portrait');
        $filename = 'Portfolio_' . preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit') . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
        return $pdf->download($filename);
    }

    /**
     * Generate one PDF for the selected unit: merge all completed assessor tools (with marks)
     * for that unit. Unit must be one of the student's class units.
     */
    public function unitsChecklistPdf(\Illuminate\Http\Request $request)
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return redirect()->route('student.progress.index')
                ->with('warning', 'No active term found. Cannot generate PDF.');
        }

        $request->validate(['unit' => 'required|exists:units,id']);

        $enrollments = $user->enrollments()
            ->whereHas('schoolClass', function ($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with([
                'schoolClass:id,name,code' => [
                    'units:id,name,code',
                ],
            ])
            ->get();

        $allowedUnitIds = $enrollments->flatMap(function ($e) {
            return ($e->schoolClass->units ?? collect())->pluck('id');
        })->unique()->values()->all();

        if (!in_array((int) $request->unit, $allowedUnitIds, true)) {
            abort(403, 'You can only download assessor tools for units in your class.');
        }

        $unit = \App\Models\Unit::findOrFail($request->unit);
        $classIds = $enrollments->pluck('class_id')->unique()->values()->all();

        $practicalAssignments = Assignment::where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->where('type', 'practical')
            ->where('is_published', true)
            ->orderBy('due_date')
            ->orderBy('id')
            ->get();

        $pdfPaths = [];
        foreach ($practicalAssignments as $assignment) {
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $user->id)
                ->first();

            if (!$submission || !$submission->completed_assessor_tool_path) {
                continue;
            }

            $path = $submission->completed_assessor_tool_path;
            if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                $pdfPaths[] = $path;
            }
        }

        $unitsWithPdfPaths = [];
        if (count($pdfPaths) > 0) {
            $unitsWithPdfPaths[] = ['unit' => $unit, 'pdf_paths' => $pdfPaths];
        }

        // No completed assessor tools for this unit: return a simple info PDF
        if (empty($unitsWithPdfPaths)) {
            $pdf = Pdf::loadView('student.progress.units-checklist-empty', [
                'user' => $user,
                'activeTerm' => $activeTerm,
                'unit' => $unit,
            ]);
            $pdf->setPaper('a4', 'portrait');
            $sanitized = preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit');
            $filename = 'Assessor_tools_' . $sanitized . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);
        }

        $tempDir = 'progress-pdf-temp/' . uniqid('', true);
        Storage::disk('local')->makeDirectory($tempDir);
        $tempPaths = [];

        try {
            $merger = new Merger();

            foreach ($unitsWithPdfPaths as $item) {
                $unit = $item['unit'];
                $paths = $item['pdf_paths'];

                // Divider page for this unit
                $dividerPdf = Pdf::loadView('student.progress.units-checklist-divider', compact('unit'));
                $dividerPdf->setPaper('a4', 'portrait');
                $dividerPath = $tempDir . '/divider_' . $unit->id . '.pdf';
                Storage::disk('local')->put($dividerPath, $dividerPdf->output());
                $tempPaths[] = Storage::disk('local')->path($dividerPath);
                $merger->addFile($tempPaths[count($tempPaths) - 1]);

                // This unit's assessor tool PDFs (with marks)
                foreach ($paths as $path) {
                    $absolutePath = Storage::disk('public')->path($path);
                    if (is_readable($absolutePath)) {
                        $merger->addFile($absolutePath);
                    }
                }
            }

            $mergedContent = $merger->merge();
        } finally {
            Storage::disk('local')->deleteDirectory($tempDir);
        }

        if (!$mergedContent) {
            return redirect()->route('student.progress.index')
                ->with('error', 'Could not generate PDF.');
        }

        $sanitized = preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit');
        $filename = 'Assessor_tools_' . $sanitized . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
        return response($mergedContent, 200, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * Generate one PDF for the selected unit's exams: merge the student's submitted exam
     * documents (PDFs from assignment submissions, non-practical) for that unit.
     */
    public function unitsExamsPdf(\Illuminate\Http\Request $request)
    {
        $user = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            return redirect()->route('student.progress.index')
                ->with('warning', 'No active term found. Cannot generate PDF.');
        }

        $request->validate(['unit' => 'required|exists:units,id']);

        $enrollments = $user->enrollments()
            ->whereHas('schoolClass', function ($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->with(['schoolClass:id,name,code' => ['units:id,name,code']])
            ->get();

        $allowedUnitIds = $enrollments->flatMap(function ($e) {
            return ($e->schoolClass->units ?? collect())->pluck('id');
        })->unique()->values()->all();

        if (!in_array((int) $request->unit, $allowedUnitIds, true)) {
            abort(403, 'You can only download exams for units in your class.');
        }

        $unit = \App\Models\Unit::findOrFail($request->unit);
        $classIds = $enrollments->pluck('class_id')->unique()->values()->all();

        $examAssignments = Assignment::where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->where('is_published', true)
            ->where('type', '!=', 'practical')
            ->orderBy('start_date')
            ->orderBy('id')
            ->get();

        $pdfPaths = [];
        foreach ($examAssignments as $assignment) {
            $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $user->id)
                ->first();

            if (!$submission || !is_array($submission->attachments)) {
                continue;
            }

            foreach ($submission->attachments as $path) {
                if (Str::endsWith(strtolower($path), '.pdf') && Storage::disk('public')->exists($path)) {
                    $pdfPaths[] = $path;
                    break;
                }
            }
        }

        if (empty($pdfPaths)) {
            $pdf = Pdf::loadView('student.progress.units-exams-empty', [
                'user' => $user,
                'activeTerm' => $activeTerm,
                'unit' => $unit,
            ]);
            $pdf->setPaper('a4', 'portrait');
            $sanitized = preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit');
            $filename = 'Exams_' . $sanitized . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
            return $pdf->download($filename);
        }

        $tempDir = 'progress-pdf-temp/' . uniqid('', true);
        Storage::disk('local')->makeDirectory($tempDir);

        try {
            $merger = new Merger();

            $dividerPdf = Pdf::loadView('student.progress.units-checklist-divider', compact('unit'));
            $dividerPdf->setPaper('a4', 'portrait');
            $dividerPath = $tempDir . '/divider_' . $unit->id . '.pdf';
            Storage::disk('local')->put($dividerPath, $dividerPdf->output());
            $merger->addFile(Storage::disk('local')->path($dividerPath));

            foreach ($pdfPaths as $path) {
                $absolutePath = Storage::disk('public')->path($path);
                if (is_readable($absolutePath)) {
                    $merger->addFile($absolutePath);
                }
            }

            $mergedContent = $merger->merge();
        } finally {
            Storage::disk('local')->deleteDirectory($tempDir);
        }

        if (!$mergedContent) {
            return redirect()->route('student.progress.index')
                ->with('error', 'Could not generate PDF.');
        }

        $sanitized = preg_replace('/[^a-z0-9]+/i', '_', $unit->name ?? 'unit');
        $filename = 'Exams_' . $sanitized . '_' . ($user->admission_number ?? 'student') . '_' . now()->format('Y-m-d') . '.pdf';
        return response($mergedContent, 200, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }
}
