<?php

namespace App\Http\Controllers\Validator;

use App\Http\Controllers\Controller;
use App\Models\ValidationRequest;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;

/**
 * ValidationRequestController
 *
 * Thin wrapper that wires the existing validator unit validation logic
 * (implemented in UnitController) to the current routes and route names
 * `validator.validation-requests.*`.
 *
 * IMPORTANT: This controller is intentionally very small and delegates all
 * heavy lifting to UnitController to avoid duplicating complex logic.
 */
class ValidationRequestController extends Controller
{
    /**
     * List validation requests for the active term.
     *
     * Route: GET /validator/validation-requests
     * Name:  validator.validation-requests.index
     */
    public function index(Request $request)
    {
        // Delegate to the existing UnitController@index implementation
        return app(UnitController::class)->index($request);
    }

    /**
     * Show validation details for a specific class + unit.
     *
     * Route: GET /validator/validation-requests/classes/{class}/units/{unit}
     * Name:  validator.validation-requests.show
     */
    public function show(SchoolClass $class, Unit $unit)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            abort(404, 'No active term found.');
        }

        // Find the validation request for this class + unit + term
        $validationRequest = ValidationRequest::where('class_id', $class->id)
            ->where('unit_id', $unit->id)
            ->where('term_id', $activeTerm->id)
            ->orderByDesc('submitted_at')
            ->firstOrFail();

        // Delegate rendering to UnitController@show, which already prepares
        // all required data and uses the rich `validator.units.show` view.
        return app(UnitController::class)->show($validationRequest);
    }

    /**
     * Unit summary marks preview/print for validator.
     *
     * Route: GET /validator/validation-requests/classes/{class}/units/{unit}/print-marks
     * Name:  validator.validation-requests.print-marks
     */
    public function showPrintMarks(SchoolClass $class, Unit $unit)
    {
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm) {
            abort(404, 'No active term found.');
        }

        $validationRequest = ValidationRequest::where('class_id', $class->id)
            ->where('unit_id', $unit->id)
            ->where('term_id', $activeTerm->id)
            ->orderByDesc('submitted_at')
            ->firstOrFail();

        return app(UnitController::class)->printMarks($validationRequest);
    }

    /**
     * Handle validation submissions.
     *
     * This endpoint serves BOTH:
     * - per-student decisions (when student_id is present), and
     * - final unit decision (when only status/comment are present).
     *
     * Route: POST /validator/validation-requests/{validationRequest}/validate
     * Name:  validator.validation-requests.validate
     */
    public function submitValidation(Request $request, ValidationRequest $validationRequest)
    {
        // If a specific student_id is present, treat this as a per-student decision
        if ($request->has('student_id')) {
            return app(UnitController::class)->updateStudentDecision($request, $validationRequest);
        }

        // Otherwise, finalize the entire unit validation
        return app(UnitController::class)->finalize($request, $validationRequest);
    }

    /**
     * View portfolio: student list for a validation request.
     *
     * Route: GET /validator/validation-requests/{validationRequest}/portfolio
     * Name:  validator.validation-requests.portfolio
     */
    public function portfolioIndex(ValidationRequest $validationRequest)
    {
        return app(UnitController::class)->portfolioIndex($validationRequest);
    }

    /**
     * View portfolio: gallery for a specific student within a validation request.
     *
     * Route: GET /validator/validation-requests/{validationRequest}/portfolio/students/{student}
     * Name:  validator.validation-requests.portfolio.student
     */
    public function portfolioStudent(ValidationRequest $validationRequest, \App\Models\User $student)
    {
        return app(UnitController::class)->portfolioStudent($validationRequest, $student);
    }

    /**
     * View portfolio: compiled exams PDF for a specific student.
     *
     * Route: GET /validator/validation-requests/{validationRequest}/portfolio/students/{student}/exams
     * Name:  validator.validation-requests.portfolio.student.exams
     */
    public function portfolioStudentExams(ValidationRequest $validationRequest, \App\Models\User $student)
    {
        return app(UnitController::class)->portfolioStudentExams($validationRequest, $student);
    }

    /**
     * View portfolio: compiled practicals PDF for a specific student.
     *
     * Route: GET /validator/validation-requests/{validationRequest}/portfolio/students/{student}/practicals
     * Name:  validator.validation-requests.portfolio.student.practicals
     */
    public function portfolioStudentPracticals(ValidationRequest $validationRequest, \App\Models\User $student)
    {
        return app(UnitController::class)->portfolioStudentPracticals($validationRequest, $student);
    }
}

