<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class PracticalEvidenceTask extends Model
{
    use HasFactory;

    protected $table = 'practical_evidence_tasks';

    protected $fillable = [
        'student_id',
        'unit_id',
        'class_id',
        'term_id',
        'task_name',
        'status',
        'trainer_comment',
        'reviewed_by',
        'submitted_at',
        'reviewed_at',
    ];

    protected $casts = [
        'submitted_at' => 'datetime',
        'reviewed_at' => 'datetime',
    ];

    // Relationships
    public function student()
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function unit()
    {
        return $this->belongsTo(Unit::class);
    }

    public function schoolClass()
    {
        return $this->belongsTo(SchoolClass::class, 'class_id');
    }

    public function term()
    {
        return $this->belongsTo(Term::class);
    }

    public function reviewer()
    {
        return $this->belongsTo(User::class, 'reviewed_by');
    }

    /**
     * Get query builder for practical evidence files for this task
     */
    public function evidenceFilesQuery()
    {
        return PracticalEvidence::where('student_id', $this->student_id)
            ->where('unit_id', $this->unit_id)
            ->where('class_id', $this->class_id)
            ->where('task_name', $this->task_name)
            ->where('term_id', $this->term_id);
    }

    /**
     * Get all practical evidence files for this task
     */
    public function evidenceFiles()
    {
        return $this->evidenceFilesQuery()
            ->orderBy('uploaded_at')
            ->get();
    }

    /**
     * Get file count for this task
     */
    public function getFileCountAttribute()
    {
        return $this->evidenceFiles()->count();
    }

    /**
     * Get status badge classes
     */
    public function getStatusBadgeClasses()
    {
        return match($this->status) {
            'submitted' => 'bg-blue-100 text-blue-800',
            'under_review' => 'bg-yellow-100 text-yellow-800',
            'accepted' => 'bg-green-100 text-green-800',
            'rejected' => 'bg-red-100 text-red-800',
            'needs_correction' => 'bg-orange-100 text-orange-800',
            default => 'bg-gray-100 text-gray-800',
        };
    }

    /**
     * Get status label
     */
    public function getStatusLabel()
    {
        return match($this->status) {
            'submitted' => 'Submitted',
            'under_review' => 'Under Review',
            'accepted' => 'Accepted',
            'rejected' => 'Rejected',
            'needs_correction' => 'Needs Correction',
            default => ucfirst($this->status),
        };
    }

    /**
     * Check if task is submitted
     */
    public function isSubmitted()
    {
        return $this->status === 'submitted';
    }

    /**
     * Check if task is accepted
     */
    public function isAccepted()
    {
        return $this->status === 'accepted';
    }

    /**
     * Check if task is rejected
     */
    public function isRejected()
    {
        return $this->status === 'rejected';
    }

    /**
     * Check if task needs correction
     */
    public function needsCorrection()
    {
        return $this->status === 'needs_correction';
    }
}
