<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Setting extends Model
{
    use HasFactory;

    protected $fillable = [
        'key',
        'value',
        'description',
        'type',
    ];

    /**
     * Get a setting value by key
     */
    public static function get($key, $default = null)
    {
        $setting = self::where('key', $key)->first();
        return $setting ? self::castValue($setting->value, $setting->type) : $default;
    }

    /**
     * Set a setting value
     */
    public static function set($key, $value, $type = 'string', $description = null)
    {
        $storedValue = is_null($value) ? '' : (string) $value;
        
        return self::updateOrCreate(
            ['key' => $key],
            [
                'value' => $storedValue,
                'type' => $type,
                'description' => $description ?? self::getDefaultDescription($key),
            ]
        );
    }

    /**
     * Cast value based on type
     */
    protected static function castValue($value, $type)
    {
        if (is_null($value) || $value === '') {
            return null;
        }

        return match ($type) {
            'integer' => (int) $value,
            'boolean' => filter_var($value, FILTER_VALIDATE_BOOLEAN),
            'json' => json_decode($value, true),
            default => $value,
        };
    }

    /**
     * Get default description for a setting key
     */
    protected static function getDefaultDescription($key)
    {
        $descriptions = [
            'institution.name' => 'Institution name',
            'institution.address' => 'Institution address',
            'institution.phone' => 'Institution phone number',
            'institution.email' => 'Institution email',
            'institution.website' => 'Institution website',
            'institution.logo' => 'Institution logo path',
            'institution.footer_text' => 'Footer text for reports',
            'system.name' => 'System name',
            'system.timezone' => 'System timezone',
            'system.date_format' => 'Date format',
            'system.time_format' => 'Time format',
        ];

        return $descriptions[$key] ?? 'System setting';
    }
}

