<?php

namespace App\Policies;

use App\Models\PoeSubmission;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class PoeSubmissionPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // Admin, HOD, Trainer, Exams Office, Validator can view all
        // Students can only view their own
        return $user->isAdmin() || $user->isHOD() || $user->isTrainer() 
            || $user->isExamsOffice() || $user->isValidator() || $user->isStudent();
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PoeSubmission  $poeSubmission
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, PoeSubmission $poeSubmission)
    {
        // Students can only view their own submissions
        if ($user->isStudent()) {
            return $user->id === $poeSubmission->student_id;
        }
        
        // Other roles can view submissions in their department or all (admin)
        if ($user->isAdmin()) {
            return true;
        }
        
        if ($user->isHOD() || $user->isTrainer()) {
            return $user->department_id === $poeSubmission->schoolClass->department_id;
        }
        
        return $user->isExamsOffice() || $user->isValidator();
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // Only students can create PoE submissions
        return $user->isStudent();
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PoeSubmission  $poeSubmission
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, PoeSubmission $poeSubmission)
    {
        // Students can update their own draft submissions
        if ($user->isStudent() && $poeSubmission->student_id === $user->id) {
            return $poeSubmission->isDraft();
        }
        
        // Admin can update any
        return $user->isAdmin();
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PoeSubmission  $poeSubmission
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, PoeSubmission $poeSubmission)
    {
        // Students can delete their own draft submissions
        if ($user->isStudent() && $poeSubmission->student_id === $user->id) {
            return $poeSubmission->isDraft();
        }
        
        // Admin can delete any
        return $user->isAdmin();
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PoeSubmission  $poeSubmission
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, PoeSubmission $poeSubmission)
    {
        return $user->isAdmin();
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\PoeSubmission  $poeSubmission
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, PoeSubmission $poeSubmission)
    {
        return $user->isAdmin();
    }
    
    /**
     * Determine whether the user can review the submission.
     */
    public function review(User $user, PoeSubmission $poeSubmission)
    {
        if ($user->isTrainer() || $user->isHOD() || $user->isExamsOffice()) {
            return $user->department_id === $poeSubmission->schoolClass->department_id;
        }
        return false;
    }
    
    /**
     * Determine whether the user can validate the submission.
     */
    public function validate(User $user, PoeSubmission $poeSubmission)
    {
        return $user->isValidator();
    }
}
