<?php

namespace App\Services;

use App\Models\Term;
use Illuminate\Support\Facades\Cache;

class ActiveTermService
{
    /**
     * Get the currently active term
     *
     * @return Term|null
     */
    public static function getActiveTerm()
    {
        return Cache::remember('active_term', 300, function () {
            return Term::where('status', 'active')->first();
        });
    }

    /**
     * Get the active term ID (optimized - no model loading)
     *
     * @return int|null
     */
    public static function getActiveTermId()
    {
        return Cache::remember('active_term_id', 300, function () {
            return Term::where('status', 'active')->value('id');
        });
    }

    /**
     * Check if there is an active term
     *
     * @return bool
     */
    public static function hasActiveTerm()
    {
        return self::getActiveTerm() !== null;
    }

    /**
     * Clear the active term cache
     */
    public static function clearCache()
    {
        Cache::forget('active_term');
        Cache::forget('active_term_id');
    }

    /**
     * Scope a query to the active term
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $termColumn
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public static function scopeToActiveTerm($query, $termColumn = 'term_id')
    {
        $activeTermId = self::getActiveTermId();
        
        if ($activeTermId) {
            return $query->where($termColumn, $activeTermId);
        }
        
        // If no active term, return empty result
        return $query->whereRaw('1 = 0');
    }

    /**
     * Require an active term or throw exception
     *
     * @return Term
     * @throws \Exception
     */
    public static function requireActiveTerm()
    {
        $term = self::getActiveTerm();
        
        if (!$term) {
            throw new \Exception('No active term is set. Please activate a term to continue.');
        }
        
        return $term;
    }
}

