<?php

namespace App\Services;

use App\Models\AssessorToolStructure;
use App\Models\AssessorToolMark;
use App\Models\AssignmentSubmission;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\View;

class AssessorToolGeneratorService
{
    /**
     * Generate completed assessor tool document
     * 
     * @param AssessorToolStructure $structure
     * @param \Illuminate\Support\Collection $marks
     * @param AssignmentSubmission $submission
     * @param string $format 'pdf' or 'html'
     * @return mixed
     */
    public function generate(AssessorToolStructure $structure, $marks, AssignmentSubmission $submission, $format = 'pdf')
    {
        $structureData = $structure->structure;
        $sections = $structureData['sections'] ?? [];
        $headerData = $structureData['header_data'] ?? [];

        // Ensure we have all relationships needed for header metadata
        $submission->loadMissing([
            'student',
            'assignment.unit.department',
            'assignment.schoolClass.term',
            'assignment.trainer',
        ]);

        $assignment = $submission->assignment;
        $unit = $assignment->unit;
        $class = $assignment->schoolClass;
        $term = $class?->term;

        // Normalise marks collection
        $marksCollection = $marks instanceof \Illuminate\Support\Collection ? $marks : collect($marks);

        // Group marks by section and also key by row identifier for fast lookup
        $marksBySection = $marksCollection->groupBy('section');
        $marksByRow = $marksCollection->keyBy('row_identifier');

        // Calculate section subtotals and overall total
        $sectionTotals = [];
        $overallTotal = 0;

        foreach ($marksBySection as $sectionName => $sectionMarks) {
            $sectionTotal = $sectionMarks->sum('marks_obtained');
            $sectionTotals[$sectionName] = $sectionTotal;
            $overallTotal += $sectionTotal;
        }

        // Institution-wide settings (logo, name, etc.)
        $institutionSettings = [
            'institution.logo' => Setting::get('institution.logo'),
            'institution.name' => Setting::get('institution.name'),
            'institution.address' => Setting::get('institution.address'),
            'institution.phone' => Setting::get('institution.phone'),
            'institution.email' => Setting::get('institution.email'),
            'institution.footer_text' => Setting::get('institution.footer_text'),
        ];

        // Prepare data for view
        $data = [
            'structure' => $structure,
            'submission' => $submission,
            'student' => $submission->student,
            'assignment' => $assignment,
            'sections' => $sections,
            'marks' => $marksCollection,
            'marksBySection' => $marksBySection,
            'marksByRow' => $marksByRow,
            'sectionTotals' => $sectionTotals,
            'overallTotal' => $overallTotal,
            'headerData' => array_merge($headerData, [
                'candidate_name' => $submission->student->name ?? '',
                'candidate_registration' => $submission->student->admission_number ?? '',
                'assessor_name' => $submission->assignment->trainer->name ?? '',
                'date' => $submission->graded_at ? $submission->graded_at->format('Y-m-d') : now()->format('Y-m-d'),
            ]),
            'institutionSettings' => $institutionSettings,
            'reportMeta' => [
                'department_name' => $unit?->department?->name ?? '',
                'course_name' => $class?->name ?? '',
                'unit_name' => $unit?->name ?? '',
                'class_code' => $class?->code ?? '',
                'unit_code' => $unit?->code ?? '',
                'series' => $term?->name ?? '',
            ],
        ];

        if ($format === 'html') {
            return View::make('trainer.assessor-tool.generated-document', $data)->render();
        }

        // Generate PDF
        $pdf = Pdf::loadView('trainer.assessor-tool.generated-document', $data);
        $pdf->setPaper('a4', 'portrait');
        
        return $pdf->download('assessor-tool-' . $submission->id . '.pdf');
    }

    /**
     * Generate and store completed assessor tool document
     * 
     * @param AssessorToolStructure $structure
     * @param \Illuminate\Support\Collection $marks
     * @param AssignmentSubmission $submission
     * @return string Path to stored document
     */
    public function generateAndStore(AssessorToolStructure $structure, $marks, AssignmentSubmission $submission)
    {
        $structureData = $structure->structure;
        $sections = $structureData['sections'] ?? [];
        $headerData = $structureData['header_data'] ?? [];

        // Ensure we have all relationships needed for header metadata
        $submission->loadMissing([
            'student',
            'assignment.unit.department',
            'assignment.schoolClass.term',
            'assignment.trainer',
        ]);

        $assignment = $submission->assignment;
        $unit = $assignment->unit;
        $class = $assignment->schoolClass;
        $term = $class?->term;

        // Normalise marks collection
        $marksCollection = $marks instanceof \Illuminate\Support\Collection ? $marks : collect($marks);

        // Group marks by section and also key by row identifier
        $marksBySection = $marksCollection->groupBy('section');
        $marksByRow = $marksCollection->keyBy('row_identifier');

        // Calculate section subtotals and overall total
        $sectionTotals = [];
        $overallTotal = 0;

        foreach ($marksBySection as $sectionName => $sectionMarks) {
            $sectionTotal = $sectionMarks->sum('marks_obtained');
            $sectionTotals[$sectionName] = $sectionTotal;
            $overallTotal += $sectionTotal;
        }

        // Institution-wide settings (logo, name, etc.)
        $institutionSettings = [
            'institution.logo' => Setting::get('institution.logo'),
            'institution.name' => Setting::get('institution.name'),
            'institution.address' => Setting::get('institution.address'),
            'institution.phone' => Setting::get('institution.phone'),
            'institution.email' => Setting::get('institution.email'),
            'institution.footer_text' => Setting::get('institution.footer_text'),
        ];

        // Prepare data for view
        $data = [
            'structure' => $structure,
            'submission' => $submission,
            'student' => $submission->student,
            'assignment' => $assignment,
            'sections' => $sections,
            'marks' => $marksCollection,
            'marksBySection' => $marksBySection,
            'marksByRow' => $marksByRow,
            'sectionTotals' => $sectionTotals,
            'overallTotal' => $overallTotal,
            'headerData' => array_merge($headerData, [
                'candidate_name' => $submission->student->name ?? '',
                'candidate_registration' => $submission->student->admission_number ?? '',
                'assessor_name' => $submission->assignment->trainer->name ?? '',
                'date' => $submission->graded_at ? $submission->graded_at->format('Y-m-d') : now()->format('Y-m-d'),
            ]),
            'institutionSettings' => $institutionSettings,
            'reportMeta' => [
                'department_name' => $unit?->department?->name ?? '',
                'course_name' => $class?->name ?? '',
                'unit_name' => $unit?->name ?? '',
                'class_code' => $class?->code ?? '',
                'unit_code' => $unit?->code ?? '',
                'series' => $term?->name ?? '',
            ],
        ];

        // Generate PDF
        $pdf = Pdf::loadView('trainer.assessor-tool.generated-document', $data);
        $pdf->setPaper('a4', 'portrait');
        
        // Store the PDF with a simple filename based on assignment title
        $assignmentTitle = $submission->assignment->title ?? 'assignment';
        $sanitizedTitle = $this->sanitizeFilename($assignmentTitle);
        $filename = 'assessor_tool_' . $sanitizedTitle . '.pdf';
        $path = 'assessor-tools/completed/' . $filename;
        
        \Illuminate\Support\Facades\Storage::disk('public')->put($path, $pdf->output());
        
        return $path;
    }

    /**
     * Sanitize a string for use in filenames
     * 
     * @param string $title
     * @return string
     */
    private function sanitizeFilename($title)
    {
        // Convert to lowercase
        $sanitized = strtolower($title);
        
        // Replace spaces and special characters with underscores
        $sanitized = preg_replace('/[^a-z0-9]+/', '_', $sanitized);
        
        // Remove leading/trailing underscores
        $sanitized = trim($sanitized, '_');
        
        // Limit length to 50 characters
        $sanitized = substr($sanitized, 0, 50);
        
        // Remove trailing underscores after truncation
        $sanitized = rtrim($sanitized, '_');
        
        // If empty after sanitization, use a default
        if (empty($sanitized)) {
            $sanitized = 'assignment';
        }
        
        return $sanitized;
    }
}

