<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class DatabaseHealthCheck
{
    /**
     * Check if database connection is available.
     *
     * @return array
     */
    public static function check(): array
    {
        try {
            // Simple query to test connection
            DB::connection()->getPdo();
            
            // Try a simple query
            DB::select('SELECT 1');
            
            return [
                'status' => 'connected',
                'message' => 'Database connection is healthy',
                'timestamp' => now()->toDateTimeString(),
            ];
        } catch (Exception $e) {
            Log::error('Database Health Check Failed', [
                'message' => $e->getMessage(),
                'code' => $e->getCode(),
            ]);

            return [
                'status' => 'disconnected',
                'message' => self::getErrorMessage($e),
                'error' => $e->getMessage(),
                'timestamp' => now()->toDateTimeString(),
            ];
        }
    }

    /**
     * Get user-friendly error message.
     *
     * @param  Exception  $e
     * @return string
     */
    protected static function getErrorMessage(Exception $e): string
    {
        $message = strtolower($e->getMessage());

        if (stripos($message, 'could not translate host name') !== false || 
            stripos($message, 'unknown host') !== false) {
            return 'Unable to resolve database hostname. Please check your internet connection.';
        }

        if (stripos($message, 'connection refused') !== false) {
            return 'Database server is not accepting connections.';
        }

        if (stripos($message, 'connection timed out') !== false || 
            stripos($message, 'timeout') !== false) {
            return 'Connection to database server timed out. Please check your internet connection.';
        }

        if (stripos($message, 'network is unreachable') !== false) {
            return 'Unable to reach database server. Please check your internet connection.';
        }

        return 'Unable to connect to database. Please check your connection and try again.';
    }

    /**
     * Check if database is available (quick check).
     *
     * @return bool
     */
    public static function isAvailable(): bool
    {
        try {
            DB::connection()->getPdo();
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}

