<?php

namespace App\Services;

/**
 * Google Analytics (GA4) Service
 * 
 * Provides structured event tracking for system usage analytics
 * without exposing sensitive academic or personal data.
 */
class GoogleAnalyticsService
{
    /**
     * GA4 Measurement ID from environment
     */
    protected static function getMeasurementId(): ?string
    {
        return config('services.ga4.measurement_id');
    }

    /**
     * Check if GA4 is enabled
     */
    public static function isEnabled(): bool
    {
        return !empty(self::getMeasurementId()) && config('services.ga4.enabled', false);
    }

    /**
     * Track page view
     * 
     * @param string $pageTitle
     * @param string $pagePath
     * @param string|null $userRole
     * @return string JavaScript code for GA4
     */
    public static function trackPageView(string $pageTitle, string $pagePath, ?string $userRole = null): string
    {
        if (!self::isEnabled()) {
            return '';
        }

        $role = $userRole ? self::sanitizeRole($userRole) : null;
        
        $js = "gtag('config', '" . self::getMeasurementId() . "', {";
        $js .= "'page_title': " . json_encode($pageTitle) . ",";
        $js .= "'page_path': " . json_encode($pagePath);
        if ($role) {
            $js .= ",'user_role': " . json_encode($role);
        }
        $js .= "});";

        return $js;
    }

    /**
     * Track custom event
     * 
     * @param string $eventName
     * @param array $eventParams
     * @return string JavaScript code for GA4
     */
    public static function trackEvent(string $eventName, array $eventParams = []): string
    {
        if (!self::isEnabled()) {
            return '';
        }

        // Sanitize event parameters to remove sensitive data
        $sanitizedParams = self::sanitizeEventParams($eventParams);

        $js = "gtag('event', " . json_encode($eventName) . ", ";
        $js .= json_encode($sanitizedParams);
        $js .= ");";

        return $js;
    }

    /**
     * Track report generation
     * 
     * @param string $reportType (e.g., 'unit_marks', 'class_marks', 'student_report')
     * @param string|null $userRole
     * @return string
     */
    public static function trackReportGeneration(string $reportType, ?string $userRole = null): string
    {
        return self::trackEvent('report_generated', [
            'report_type' => $reportType,
            'user_role' => $userRole ? self::sanitizeRole($userRole) : null,
        ]);
    }

    /**
     * Track submission event
     * 
     * @param string $action (e.g., 'submitted', 'reviewed', 'approved', 'rejected')
     * @param string|null $userRole
     * @return string
     */
    public static function trackSubmission(string $action, ?string $userRole = null): string
    {
        return self::trackEvent('submission_action', [
            'action' => $action,
            'user_role' => $userRole ? self::sanitizeRole($userRole) : null,
        ]);
    }

    /**
     * Track feature interaction
     * 
     * @param string $featureName
     * @param string $interactionType (e.g., 'viewed', 'clicked', 'downloaded')
     * @param string|null $userRole
     * @return string
     */
    public static function trackFeatureInteraction(string $featureName, string $interactionType, ?string $userRole = null): string
    {
        return self::trackEvent('feature_interaction', [
            'feature_name' => $featureName,
            'interaction_type' => $interactionType,
            'user_role' => $userRole ? self::sanitizeRole($userRole) : null,
        ]);
    }

    /**
     * Track navigation pattern
     * 
     * @param string $fromPage
     * @param string $toPage
     * @param string|null $userRole
     * @return string
     */
    public static function trackNavigation(string $fromPage, string $toPage, ?string $userRole = null): string
    {
        return self::trackEvent('navigation', [
            'from_page' => self::sanitizePageName($fromPage),
            'to_page' => self::sanitizePageName($toPage),
            'user_role' => $userRole ? self::sanitizeRole($userRole) : null,
        ]);
    }

    /**
     * Track dashboard view
     * 
     * @param string $dashboardType (e.g., 'hod', 'trainer', 'student', 'validator')
     * @return string
     */
    public static function trackDashboardView(string $dashboardType): string
    {
        return self::trackEvent('dashboard_viewed', [
            'dashboard_type' => $dashboardType,
        ]);
    }

    /**
     * Track marks management action
     * 
     * @param string $action (e.g., 'viewed', 'printed', 'exported')
     * @param string|null $userRole
     * @return string
     */
    public static function trackMarksAction(string $action, ?string $userRole = null): string
    {
        return self::trackEvent('marks_action', [
            'action' => $action,
            'user_role' => $userRole ? self::sanitizeRole($userRole) : null,
        ]);
    }

    /**
     * Get GA4 initialization script
     * 
     * @return string
     */
    public static function getInitScript(): string
    {
        if (!self::isEnabled()) {
            return '';
        }

        $measurementId = self::getMeasurementId();
        
        return <<<HTML
<!-- Google tag (gtag.js) -->
<script async src="https://www.googletagmanager.com/gtag/js?id={$measurementId}"></script>
<script>
  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());
  gtag('config', '{$measurementId}', {
    'anonymize_ip': true,
    'cookie_flags': 'SameSite=None;Secure'
  });
</script>
HTML;
    }

    /**
     * Sanitize event parameters to remove sensitive data
     * 
     * @param array $params
     * @return array
     */
    protected static function sanitizeEventParams(array $params): array
    {
        $sensitiveKeys = [
            'student_id', 'student_name', 'admission_number', 'email', 'phone',
            'marks', 'scores', 'grades', 'personal_data', 'academic_data',
            'submission_id', 'unit_id', 'class_id', 'term_id',
        ];

        $sanitized = [];
        foreach ($params as $key => $value) {
            // Skip sensitive keys
            if (in_array(strtolower($key), $sensitiveKeys)) {
                continue;
            }

            // Skip if value contains sensitive patterns
            if (is_string($value) && self::containsSensitiveData($value)) {
                continue;
            }

            $sanitized[$key] = $value;
        }

        return $sanitized;
    }

    /**
     * Check if string contains sensitive data patterns
     * 
     * @param string $value
     * @return bool
     */
    protected static function containsSensitiveData(string $value): bool
    {
        // Patterns that might indicate sensitive data
        $patterns = [
            '/\d{4,}/', // Long numbers (might be IDs)
            '/[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}/i', // Email pattern
            '/\+?\d{10,}/', // Phone number pattern
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $value)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Sanitize user role for analytics
     * 
     * @param string $role
     * @return string
     */
    protected static function sanitizeRole(string $role): string
    {
        $allowedRoles = ['admin', 'hod', 'trainer', 'student', 'validator'];
        $role = strtolower($role);
        
        return in_array($role, $allowedRoles) ? $role : 'unknown';
    }

    /**
     * Sanitize page name for analytics
     * 
     * @param string $pageName
     * @return string
     */
    protected static function sanitizePageName(string $pageName): string
    {
        // Remove IDs and sensitive identifiers
        $pageName = preg_replace('/\/\d+/', '/[id]', $pageName);
        $pageName = preg_replace('/\?.*$/', '', $pageName); // Remove query strings
        
        return $pageName;
    }
}

