@extends('layouts.hod')

@section('title', 'HOD Dashboard')
@section('page-title', 'Dashboard')

@section('content')
<div class="space-y-6">
    <!-- Welcome Card with Action Buttons -->
    <div class="bg-gradient-to-r from-purple-600 via-purple-700 to-purple-800 rounded-xl shadow-xl p-6 md:p-8 text-white overflow-hidden relative">
        <!-- Decorative background elements -->
        <div class="absolute top-0 right-0 -mt-4 -mr-4 w-64 h-64 bg-white opacity-5 rounded-full"></div>
        <div class="absolute bottom-0 left-0 -mb-8 -ml-8 w-48 h-48 bg-white opacity-5 rounded-full"></div>
        
        <div class="relative z-10 flex flex-col md:flex-row md:items-center md:justify-between">
            <div class="flex-1 mb-4 md:mb-0">
                <h1 class="text-3xl md:text-4xl font-bold mb-2">Welcome, {{ auth()->user()->name }}!</h1>
                <div class="flex flex-wrap items-center gap-3 mt-3">
                    @if($department)
                        <div class="flex items-center text-purple-100">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"></path>
                            </svg>
                            <span class="text-sm font-medium">{{ $department->name }} Department</span>
                        </div>
                    @endif
                    @if($activeTerm)
                        <span class="inline-flex items-center px-4 py-2 rounded-full text-sm font-semibold bg-white bg-opacity-20 text-white border border-white border-opacity-30 backdrop-blur-sm">
                            <svg class="w-4 h-4 mr-1.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"></path>
                            </svg>
                            Current Term: {{ $activeTerm->name }}
                        </span>
                    @endif
                </div>
            </div>
        </div>
    </div>

    @if(!$activeTerm)
        <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded-lg">
            <div class="flex items-center">
                <svg class="w-5 h-5 text-yellow-600 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                </svg>
                <p class="text-yellow-800 font-medium">No active term found. Please contact your administrator.</p>
            </div>
        </div>
    @elseif(!$department)
        <div class="bg-yellow-50 border-l-4 border-yellow-400 p-4 rounded-lg">
            <div class="flex items-center">
                <svg class="w-5 h-5 text-yellow-600 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                </svg>
                <p class="text-yellow-800 font-medium">You are not assigned to a department. Please contact your administrator.</p>
            </div>
        </div>
    @else
        <!-- Department Health Indicator -->
        @if(isset($departmentHealth))
            <div class="bg-white rounded-xl shadow-md p-4 border-l-4 {{ $departmentHealth['status'] === 'good' ? 'border-green-500' : ($departmentHealth['status'] === 'attention' ? 'border-yellow-500' : 'border-red-500') }}">
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        @if($departmentHealth['status'] === 'good')
                            <svg class="w-6 h-6 text-green-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            <div>
                                <p class="font-semibold text-gray-900">Department Health: <span class="text-green-600">✅ Good</span></p>
                                <p class="text-sm text-gray-600 mt-1">All systems operational</p>
                            </div>
                        @elseif($departmentHealth['status'] === 'attention')
                            <svg class="w-6 h-6 text-yellow-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                            </svg>
                            <div>
                                <p class="font-semibold text-gray-900">Department Health: <span class="text-yellow-600">⚠️ Attention Needed</span></p>
                                <p class="text-sm text-gray-600 mt-1">{{ implode(', ', $departmentHealth['issues']) }}</p>
                            </div>
                        @else
                            <svg class="w-6 h-6 text-red-600 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                            </svg>
                            <div>
                                <p class="font-semibold text-gray-900">Department Health: <span class="text-red-600">❌ Critical</span></p>
                                <p class="text-sm text-gray-600 mt-1">{{ implode(', ', $departmentHealth['issues']) }}</p>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        @endif

        <!-- Decision Center - Primary Actions -->
        <div class="bg-white rounded-xl shadow-md p-6">
            <h2 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                <svg class="w-5 h-5 mr-2 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"></path>
                </svg>
                Decision Center
            </h2>
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <!-- Pending Approvals -->
                <a href="{{ route('hod.approvals.index', ['status' => 'pending']) }}" 
                   class="group relative overflow-hidden bg-gradient-to-br from-red-50 to-red-100 border-2 border-red-300 rounded-xl p-6 hover:border-red-400 hover:shadow-lg transition-all">
                    <div class="relative z-10">
                        <div class="flex items-center justify-between mb-3">
                            <div class="h-12 w-12 bg-red-500 rounded-lg flex items-center justify-center">
                                <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                </svg>
                            </div>
                            @if(isset($stats['pending_approvals']) && $stats['pending_approvals'] > 0)
                                <span class="bg-red-600 text-white text-xs font-bold px-3 py-1 rounded-full">{{ $stats['pending_approvals'] }}</span>
                            @endif
                        </div>
                        <h3 class="text-lg font-bold text-gray-900 mb-1">Pending Approvals</h3>
                        <p class="text-sm text-gray-600">Review submissions awaiting your decision</p>
                        <div class="mt-3 flex items-center text-red-600 font-semibold text-sm">
                            Review Now
                            <svg class="w-4 h-4 ml-1 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                            </svg>
                        </div>
                    </div>
                </a>

                <!-- Units Awaiting Allocation -->
                <a href="{{ route('hod.unit-allocations.index') }}" 
                   class="group relative overflow-hidden bg-gradient-to-br from-amber-50 to-amber-100 border-2 border-amber-300 rounded-xl p-6 hover:border-amber-400 hover:shadow-lg transition-all">
                    <div class="relative z-10">
                        <div class="flex items-center justify-between mb-3">
                            <div class="h-12 w-12 bg-amber-500 rounded-lg flex items-center justify-center">
                                <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"></path>
                                </svg>
                            </div>
                            @if(isset($departmentStats['units_awaiting_allocation']) && $departmentStats['units_awaiting_allocation'] > 0)
                                <span class="bg-amber-600 text-white text-xs font-bold px-3 py-1 rounded-full">{{ $departmentStats['units_awaiting_allocation'] }}</span>
                            @endif
                        </div>
                        <h3 class="text-lg font-bold text-gray-900 mb-1">Units Awaiting Allocation</h3>
                        <p class="text-sm text-gray-600">Assign trainers to units and classes</p>
                        <div class="mt-3 flex items-center text-amber-600 font-semibold text-sm">
                            Allocate Now
                            <svg class="w-4 h-4 ml-1 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                            </svg>
                        </div>
                    </div>
                </a>

                <!-- Marks Pending Verification -->
                <a href="{{ route('hod.marks.index') }}" 
                   class="group relative overflow-hidden bg-gradient-to-br from-green-50 to-green-100 border-2 border-green-300 rounded-xl p-6 hover:border-green-400 hover:shadow-lg transition-all">
                    <div class="relative z-10">
                        <div class="flex items-center justify-between mb-3">
                            <div class="h-12 w-12 bg-green-500 rounded-lg flex items-center justify-center">
                                <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4M7.835 4.697a3.42 3.42 0 001.946-.806 3.42 3.42 0 014.438 0 3.42 3.42 0 001.946.806 3.42 3.42 0 013.138 3.138 3.42 3.42 0 00.806 1.946 3.42 3.42 0 010 4.438 3.42 3.42 0 00-.806 1.946 3.42 3.42 0 01-3.138 3.138 3.42 3.42 0 00-1.946.806 3.42 3.42 0 01-4.438 0 3.42 3.42 0 00-1.946-.806 3.42 3.42 0 01-3.138-3.138 3.42 3.42 0 00-.806-1.946 3.42 3.42 0 010-4.438 3.42 3.42 0 00.806-1.946 3.42 3.42 0 013.138-3.138z"></path>
                                </svg>
                            </div>
                            @if(isset($marksPendingVerification) && $marksPendingVerification > 0)
                                <span class="bg-green-600 text-white text-xs font-bold px-3 py-1 rounded-full">{{ $marksPendingVerification }}</span>
                            @endif
                        </div>
                        <h3 class="text-lg font-bold text-gray-900 mb-1">Marks Pending Verification</h3>
                        <p class="text-sm text-gray-600">Review trainer-assigned marks</p>
                        <div class="mt-3 flex items-center text-green-600 font-semibold text-sm">
                            Verify Now
                            <svg class="w-4 h-4 ml-1 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                            </svg>
                        </div>
                    </div>
                </a>
            </div>
        </div>

        <!-- Statistics Cards - Neutral Design -->
        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
            <!-- Pending Approvals -->
            <a href="{{ route('hod.approvals.index', ['status' => 'pending']) }}" 
               class="bg-white rounded-xl shadow-md hover:shadow-xl transition-all duration-300 p-6 border border-gray-200 hover:border-purple-300 group">
                <div class="flex items-center justify-between">
                    <div class="flex-1">
                        <p class="text-sm font-medium text-gray-600 mb-1 flex items-center">
                            <svg class="w-4 h-4 mr-1.5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            Pending Approvals
                            <span class="ml-2 text-gray-400" title="Submissions reviewed by trainers awaiting HOD approval">
                                <svg class="w-3 h-3 inline" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                </svg>
                            </span>
                        </p>
                        <p class="text-4xl font-bold text-gray-900 mt-2" data-count="{{ $stats['pending_approvals'] ?? 0 }}">0</p>
                        @if(isset($stats['pending_approvals']) && $stats['pending_approvals'] > 0)
                            <p class="text-xs text-gray-500 mt-1">Requires your attention</p>
                        @else
                            <p class="text-xs text-green-600 mt-1">All caught up!</p>
                        @endif
                    </div>
                </div>
            </a>

            <!-- Total Submissions -->
            <a href="{{ route('hod.submissions.index') }}" 
               class="bg-white rounded-xl shadow-md hover:shadow-xl transition-all duration-300 p-6 border border-gray-200 hover:border-blue-300 group">
                <div class="flex items-center justify-between">
                    <div class="flex-1">
                        <p class="text-sm font-medium text-gray-600 mb-1 flex items-center">
                            <svg class="w-4 h-4 mr-1.5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
                            </svg>
                            Total Submissions
                        </p>
                        <p class="text-4xl font-bold text-gray-900 mt-2" data-count="{{ $stats['total_submissions'] ?? 0 }}">0</p>
                        <p class="text-xs text-gray-500 mt-1">This term</p>
                    </div>
                </div>
            </a>

            <!-- Approved -->
            <a href="{{ route('hod.approvals.index', ['status' => 'hod_approved']) }}" 
               class="bg-white rounded-xl shadow-md hover:shadow-xl transition-all duration-300 p-6 border border-gray-200 hover:border-green-300 group">
                <div class="flex items-center justify-between">
                    <div class="flex-1">
                        <p class="text-sm font-medium text-gray-600 mb-1 flex items-center">
                            <svg class="w-4 h-4 mr-1.5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                            </svg>
                            Approved
                            <span class="ml-2 text-gray-400" title="Submissions approved by HOD">
                                <svg class="w-3 h-3 inline" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                </svg>
                            </span>
                        </p>
                        <p class="text-4xl font-bold text-gray-900 mt-2" data-count="{{ $stats['approved'] ?? 0 }}">0</p>
                        @if(isset($stats['total_submissions']) && $stats['total_submissions'] > 0)
                            <p class="text-xs text-gray-500 mt-1">{{ round(($stats['approved'] / $stats['total_submissions']) * 100, 1) }}% of total</p>
                        @endif
                    </div>
                </div>
            </a>

            <!-- Under Review -->
            <a href="{{ route('hod.submissions.index', ['status' => 'under_review']) }}" 
               class="bg-white rounded-xl shadow-md hover:shadow-xl transition-all duration-300 p-6 border border-gray-200 hover:border-amber-300 group">
                <div class="flex items-center justify-between">
                    <div class="flex-1">
                        <p class="text-sm font-medium text-gray-600 mb-1 flex items-center">
                            <svg class="w-4 h-4 mr-1.5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            Under Review
                        </p>
                        <p class="text-4xl font-bold text-gray-900 mt-2" data-count="{{ $stats['under_review'] ?? 0 }}">0</p>
                        <p class="text-xs text-gray-500 mt-1">In progress</p>
                    </div>
                </div>
            </a>
        </div>

        <!-- Two Column Layout: Department Overview + Recent Activity -->
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
            <!-- Department Overview - Actionable Cards -->
            <div class="lg:col-span-2 bg-white rounded-xl shadow-md p-6">
                <div class="flex items-center justify-between mb-6">
                    <h2 class="text-xl font-bold text-gray-900 flex items-center">
                        <svg class="w-6 h-6 mr-2 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"></path>
                        </svg>
                        Department Overview
                    </h2>
                </div>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <!-- Classes Card -->
                    <a href="{{ route('hod.units.index') }}" class="bg-gradient-to-br from-gray-50 to-gray-100 rounded-xl p-5 border border-gray-200 hover:border-purple-300 hover:shadow-md transition-all group">
                        <div class="flex items-center justify-between mb-3">
                            <div class="h-10 w-10 bg-gray-600 rounded-lg flex items-center justify-center">
                                <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                                </svg>
                            </div>
                            @if(isset($departmentStats['classes_missing_trainers']) && $departmentStats['classes_missing_trainers'] > 0)
                                <span class="bg-yellow-100 text-yellow-800 text-xs font-semibold px-2 py-1 rounded-full">Action Needed</span>
                            @endif
                        </div>
                        <p class="text-3xl font-bold text-gray-900 mb-1">{{ $departmentStats['total_classes'] ?? 0 }}</p>
                        <p class="text-sm font-medium text-gray-700 mb-2">Classes</p>
                        <div class="text-xs text-gray-600">
                            <span class="text-green-600 font-medium">{{ $departmentStats['classes_with_trainers'] ?? 0 }} Active</span>
                            @if(isset($departmentStats['classes_missing_trainers']) && $departmentStats['classes_missing_trainers'] > 0)
                                <span class="text-red-600 font-medium">| {{ $departmentStats['classes_missing_trainers'] }} Missing Trainer</span>
                            @endif
                        </div>
                    </a>

                    <!-- Students Card -->
                    <div class="bg-gradient-to-br from-gray-50 to-gray-100 rounded-xl p-5 border border-gray-200">
                        <div class="flex items-center justify-between mb-3">
                            <div class="h-10 w-10 bg-gray-600 rounded-lg flex items-center justify-center">
                                <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"></path>
                                </svg>
                            </div>
                        </div>
                        <p class="text-3xl font-bold text-gray-900 mb-1">{{ $departmentStats['total_students'] ?? 0 }}</p>
                        <p class="text-sm font-medium text-gray-700 mb-2">Students</p>
                        @if(isset($departmentStats['submission_completion_rate']))
                            <div class="mt-3">
                                <div class="flex items-center justify-between text-xs mb-1">
                                    <span class="text-gray-600">Submissions</span>
                                    <span class="font-semibold text-gray-900">{{ $departmentStats['submission_completion_rate'] }}%</span>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-2">
                                    <div class="bg-purple-600 h-2 rounded-full transition-all duration-500" style="width: {{ $departmentStats['submission_completion_rate'] }}%"></div>
                                </div>
                            </div>
                        @endif
                    </div>

                    <!-- Trainers Card -->
                    <a href="{{ route('hod.unit-allocations.index') }}" class="bg-gradient-to-br from-gray-50 to-gray-100 rounded-xl p-5 border border-gray-200 hover:border-purple-300 hover:shadow-md transition-all group">
                        <div class="flex items-center justify-between mb-3">
                            <div class="h-10 w-10 bg-gray-600 rounded-lg flex items-center justify-center">
                                <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 13.255A23.931 23.931 0 0112 15c-3.183 0-6.22-.62-9-1.745M16 6V4a2 2 0 00-2-2h-4a2 2 0 00-2 2v2m4 6h.01M5 20h14a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"></path>
                                </svg>
                            </div>
                            @if(isset($departmentStats['units_awaiting_allocation']) && $departmentStats['units_awaiting_allocation'] > 0)
                                <span class="bg-yellow-100 text-yellow-800 text-xs font-semibold px-2 py-1 rounded-full">Action Needed</span>
                            @endif
                        </div>
                        <p class="text-3xl font-bold text-gray-900 mb-1">{{ $departmentStats['total_trainers'] ?? 0 }}</p>
                        <p class="text-sm font-medium text-gray-700 mb-2">Trainers</p>
                        <div class="text-xs text-gray-600">
                            <span class="text-green-600 font-medium">{{ $departmentStats['total_trainers'] ?? 0 }} Active</span>
                            @if(isset($departmentStats['units_awaiting_allocation']) && $departmentStats['units_awaiting_allocation'] > 0)
                                <span class="text-red-600 font-medium">| {{ $departmentStats['units_awaiting_allocation'] }} Unassigned Units</span>
                            @endif
                        </div>
                    </a>
                </div>
            </div>

            <!-- Recent Activity Panel -->
            <div class="bg-white rounded-xl shadow-md p-6">
                <h2 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                    <svg class="w-5 h-5 mr-2 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    Recent Activity
                </h2>
                @if(isset($recentActivity) && count($recentActivity) > 0)
                    <div class="space-y-3 max-h-96 overflow-y-auto">
                        @foreach($recentActivity as $activity)
                            <div class="flex items-start gap-3 p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                                <div class="flex-shrink-0 mt-0.5">
                                    @if($activity['icon'] === 'document')
                                        <div class="h-8 w-8 bg-blue-100 rounded-full flex items-center justify-center">
                                            <svg class="w-4 h-4 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                            </svg>
                                        </div>
                                    @elseif($activity['icon'] === 'check')
                                        <div class="h-8 w-8 bg-green-100 rounded-full flex items-center justify-center">
                                            <svg class="w-4 h-4 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                            </svg>
                                        </div>
                                    @else
                                        <div class="h-8 w-8 bg-purple-100 rounded-full flex items-center justify-center">
                                            <svg class="w-4 h-4 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.828 10.172a4 4 0 00-5.656 0l-4 4a4 4 0 105.656 5.656l1.102-1.101m-.758-4.899a4 4 0 005.656 0l4-4a4 4 0 00-5.656-5.656l-1.1 1.1"></path>
                                            </svg>
                                        </div>
                                    @endif
                                </div>
                                <div class="flex-1 min-w-0">
                                    <p class="text-sm text-gray-900">{{ $activity['message'] }}</p>
                                    <p class="text-xs text-gray-500 mt-1">{{ \Carbon\Carbon::parse($activity['time'])->diffForHumans() }}</p>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <p class="text-sm text-gray-500 text-center py-8">No recent activity</p>
                @endif
            </div>
        </div>

        <!-- Pending Approvals Section -->
        @if($pendingApprovals->count() > 0)
            <div class="bg-white rounded-xl shadow-md overflow-hidden">
                <div class="px-6 py-4 bg-gradient-to-r from-purple-50 to-purple-100 border-b border-purple-200">
                    <div class="flex items-center justify-between">
                        <h2 class="text-lg font-bold text-gray-900 flex items-center">
                            <svg class="w-5 h-5 mr-2 text-purple-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                            </svg>
                            Pending Approvals
                        </h2>
                        <a href="{{ route('hod.approvals.index') }}" class="text-sm text-purple-600 hover:text-purple-800 font-semibold flex items-center">
                            View All
                            <svg class="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                            </svg>
                        </a>
                    </div>
                </div>
                <div class="p-6">
                    <div class="space-y-3">
                        @foreach($pendingApprovals as $submission)
                            <div class="border border-gray-200 rounded-lg p-5 hover:bg-purple-50 hover:border-purple-300 transition-all duration-200 group">
                                <div class="flex items-start justify-between">
                                    <div class="flex-1">
                                        <div class="flex items-center gap-3 mb-3">
                                            <div class="h-10 w-10 rounded-full bg-purple-100 flex items-center justify-center">
                                                <span class="text-purple-600 font-semibold text-sm">{{ substr($submission->student->name, 0, 1) }}</span>
                                            </div>
                                            <div>
                                                <h3 class="font-semibold text-gray-900">{{ $submission->student->name }}</h3>
                                                <p class="text-xs text-gray-500">{{ $submission->student->email }}</p>
                                            </div>
                                        </div>
                                        <div class="flex flex-wrap items-center gap-2 mb-2">
                                            <span class="px-3 py-1 text-xs font-medium rounded-full bg-purple-100 text-purple-800 border border-purple-200">
                                                {{ $submission->unit->name }}
                                            </span>
                                            <span class="px-3 py-1 text-xs font-medium rounded-full bg-gray-100 text-gray-800 border border-gray-200">
                                                {{ $submission->schoolClass->name }}
                                            </span>
                                        </div>
                                        @if(isset($submission->days_since_review))
                                            <div class="flex items-center gap-2 text-sm">
                                                @if($submission->is_overdue)
                                                    <span class="inline-flex items-center px-2 py-1 rounded-md text-xs font-semibold bg-red-100 text-red-800 border border-red-200">
                                                        <svg class="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                                                        </svg>
                                                        {{ $submission->days_since_review }} days overdue
                                                    </span>
                                                @else
                                                    <span class="text-gray-600">
                                                        <svg class="w-4 h-4 inline mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                                                        </svg>
                                                        Reviewed {{ $submission->days_since_review }} day(s) ago
                                                    </span>
                                                @endif
                                            </div>
                                        @endif
                                    </div>
                                    <a href="{{ route('hod.approvals.show', $submission) }}" 
                                       class="ml-4 inline-flex items-center px-5 py-2.5 bg-purple-600 text-white text-sm font-semibold rounded-lg hover:bg-purple-700 shadow-md hover:shadow-lg transition-all group-hover:scale-105">
                                        Review
                                        <svg class="w-4 h-4 ml-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"></path>
                                        </svg>
                                    </a>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>
        @else
            <div class="bg-white rounded-xl shadow-md p-12">
                <div class="text-center">
                    <div class="inline-flex items-center justify-center w-20 h-20 bg-green-100 rounded-full mb-4">
                        <svg class="w-10 h-10 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                        </svg>
                    </div>
                    <h3 class="text-lg font-semibold text-gray-900 mb-2">All caught up!</h3>
                    <p class="text-sm text-gray-500 mb-4">There are no pending approvals at this time.</p>
                    <div class="flex flex-col sm:flex-row gap-2 justify-center">
                        <a href="{{ route('hod.reports.index') }}" class="inline-flex items-center px-4 py-2 text-sm font-medium text-purple-600 hover:text-purple-800">
                            <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 17v-2m3 2v-4m3 4v-6m2 10H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                            </svg>
                            Review Past Reports
                        </a>
                        <span class="text-gray-300">|</span>
                        <a href="{{ route('hod.unit-allocations.index') }}" class="inline-flex items-center px-4 py-2 text-sm font-medium text-purple-600 hover:text-purple-800">
                            <svg class="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2m-6 9l2 2 4-4"></path>
                            </svg>
                            Allocate Upcoming Units
                        </a>
                    </div>
                </div>
            </div>
        @endif

        <!-- Department Progress by Class -->
        @if($departmentProgress->count() > 0)
            <div class="bg-white rounded-xl shadow-md overflow-hidden">
                <div class="px-6 py-4 bg-gradient-to-r from-gray-50 to-gray-100 border-b border-gray-200">
                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                        <svg class="w-5 h-5 mr-2 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z"></path>
                        </svg>
                        Department Progress by Class
                    </h2>
                </div>
                <div class="p-6">
                    <div class="space-y-4">
                        @foreach($departmentProgress as $progress)
                            <div class="border border-gray-200 rounded-xl p-5 hover:border-purple-300 hover:shadow-md transition-all bg-white">
                                <div class="flex items-center justify-between mb-4">
                                    <div class="flex items-center">
                                        <div class="h-12 w-12 rounded-lg bg-gray-100 flex items-center justify-center mr-3">
                                            <svg class="w-6 h-6 text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 4.354a4 4 0 110 5.292M15 21H3v-1a6 6 0 0112 0v1zm0 0h6v-1a6 6 0 00-9-5.197M13 7a4 4 0 11-8 0 4 4 0 018 0z"></path>
                                            </svg>
                                        </div>
                                        <div>
                                            <h3 class="font-bold text-gray-900">{{ $progress['class']->name }}</h3>
                                            @if($progress['class']->code)
                                                <p class="text-xs text-gray-500">{{ $progress['class']->code }}</p>
                                            @endif
                                        </div>
                                    </div>
                                    <div class="text-right">
                                        <p class="text-2xl font-bold text-gray-900">{{ $progress['completion_rate'] }}%</p>
                                        <p class="text-xs text-gray-500">Complete</p>
                                    </div>
                                </div>
                                <div class="w-full bg-gray-200 rounded-full h-3 mb-3 overflow-hidden">
                                    <div class="bg-purple-600 h-3 rounded-full transition-all duration-500" style="width: {{ $progress['completion_rate'] }}%"></div>
                                </div>
                                <div class="flex flex-wrap gap-4 text-sm">
                                    <div class="flex items-center">
                                        <span class="w-2 h-2 bg-blue-500 rounded-full mr-2"></span>
                                        <span class="text-gray-600">Total: <strong class="text-gray-900">{{ $progress['total_submissions'] }}</strong></span>
                                    </div>
                                    <div class="flex items-center">
                                        <span class="w-2 h-2 bg-green-500 rounded-full mr-2"></span>
                                        <span class="text-gray-600">Approved: <strong class="text-gray-900">{{ $progress['approved'] }}</strong></span>
                                    </div>
                                    <div class="flex items-center">
                                        <span class="w-2 h-2 bg-yellow-500 rounded-full mr-2"></span>
                                        <span class="text-gray-600">Pending: <strong class="text-gray-900">{{ $progress['pending_approval'] }}</strong></span>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
            </div>
        @endif
    @endif
</div>

<!-- Sticky Review Approvals Button -->
@if(isset($stats['pending_approvals']) && $stats['pending_approvals'] > 0)
    <div class="fixed bottom-6 right-6 z-50 hidden md:block" id="sticky-review-button">
        <a href="{{ route('hod.approvals.index', ['status' => 'pending']) }}" 
           class="inline-flex items-center px-6 py-3 bg-purple-600 text-white font-semibold rounded-full shadow-lg hover:bg-purple-700 hover:shadow-xl transition-all animate-pulse">
            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
            Review {{ $stats['pending_approvals'] }} Approval{{ $stats['pending_approvals'] > 1 ? 's' : '' }}
        </a>
    </div>
@endif

@push('scripts')
<script>
// Number count-up animation
document.addEventListener('DOMContentLoaded', function() {
    const counters = document.querySelectorAll('[data-count]');
    
    counters.forEach(counter => {
        const target = parseInt(counter.getAttribute('data-count'));
        const duration = 1000; // 1 second
        const increment = target / (duration / 16); // 60fps
        let current = 0;
        
        const updateCounter = () => {
            current += increment;
            if (current < target) {
                counter.textContent = Math.floor(current);
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target;
            }
        };
        
        updateCounter();
    });
});

// Show sticky button on scroll
let lastScrollTop = 0;
const stickyButton = document.getElementById('sticky-review-button');
if (stickyButton) {
    window.addEventListener('scroll', function() {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        if (scrollTop > 300) {
            stickyButton.classList.remove('hidden');
        } else {
            stickyButton.classList.add('hidden');
        }
        lastScrollTop = scrollTop;
    });
}
</script>
@endpush
@endsection
