# Database Connection Error Handling

## Overview

The application now includes comprehensive error handling for database connection failures, particularly when internet connectivity is unavailable or the Supabase database cannot be reached.

## Features

### 1. Custom Exception Handler

The `App\Exceptions\Handler` class now:
- Detects database connection errors automatically
- Categorizes errors (DNS resolution, timeout, connection refused, etc.)
- Provides user-friendly error messages
- Logs errors for debugging
- Returns appropriate HTTP status codes (503 Service Unavailable)

### 2. User-Friendly Error Page

A custom error page (`resources/views/errors/database-connection.blade.php`) displays:
- Clear error message
- Helpful troubleshooting steps
- Retry button
- Auto-retry functionality (attempts to reconnect every 5 seconds)
- Debug information (only in development mode)

### 3. Database Health Check Service

The `App\Services\DatabaseHealthCheck` class provides:
- `check()` - Full health check with detailed status
- `isAvailable()` - Quick boolean check
- User-friendly error messages

### 4. Health Check Endpoint

A health check endpoint is available at `/health/database`:
- Returns JSON with connection status
- Can be used for monitoring
- Returns 200 if connected, 503 if disconnected

## Error Types Detected

1. **DNS Resolution Errors**
   - "could not translate host name"
   - "unknown host"
   - Error code: 08006

2. **Connection Refused**
   - "connection refused"
   - Error code: 08001

3. **Timeout Errors**
   - "connection timed out"
   - "timeout"
   - Error code: HY000

4. **Network Unreachable**
   - "network is unreachable"
   - "no route to host"

## Usage

### Manual Health Check

```php
use App\Services\DatabaseHealthCheck;

// Full check
$health = DatabaseHealthCheck::check();
// Returns: ['status' => 'connected'|'disconnected', 'message' => ..., ...]

// Quick check
if (DatabaseHealthCheck::isAvailable()) {
    // Database is available
}
```

### Health Check Endpoint

```bash
# Check database status
curl http://your-app/health/database

# Response when connected:
{
    "status": "connected",
    "message": "Database connection is healthy",
    "timestamp": "2025-12-31 18:00:00"
}

# Response when disconnected:
{
    "status": "disconnected",
    "message": "Unable to resolve database hostname...",
    "error": "...",
    "timestamp": "2025-12-31 18:00:00"
}
```

## Error Page Features

### Auto-Retry
- Automatically attempts to reconnect every 5 seconds
- Maximum 3 retry attempts
- Stops retrying after max attempts

### User Actions
- **Retry Connection** button - Manually retry
- **Go to Homepage** button - Navigate away
- Helpful troubleshooting tips displayed

## Configuration

### Development Mode
- Full error details shown
- Errors logged to `storage/logs/laravel.log`
- Stack traces available

### Production Mode
- User-friendly messages only
- No sensitive error details exposed
- Errors still logged for monitoring

## Testing

To test the error handling:

1. **Disconnect from internet**
2. **Try to access any page** - Should show error page
3. **Reconnect internet** - Auto-retry should work
4. **Check logs** - Error should be logged

## Monitoring

Use the health check endpoint for monitoring:
- Set up uptime monitoring to check `/health/database`
- Alert when status is "disconnected"
- Monitor response times

## Best Practices

1. **Don't cache database errors** - Always check fresh
2. **Log all connection errors** - Helps identify patterns
3. **Provide clear user feedback** - Users should know what's happening
4. **Auto-retry with limits** - Prevents infinite loops
5. **Graceful degradation** - Show error page instead of crashing

## Troubleshooting

If error handling isn't working:

1. Check `app/Exceptions/Handler.php` is being used
2. Verify error view exists: `resources/views/errors/database-connection.blade.php`
3. Check logs: `storage/logs/laravel.log`
4. Clear cache: `php artisan config:clear`

