# Quick Performance Fixes for Supabase

## ⚡ Immediate Actions (Do These First)

### 1. Use Supabase Connection Pooler

**Update your `.env` file:**

```env
# Change from port 5432 to 6543 (connection pooler)
DB_PORT=6543
```

**Why**: Connection pooling reduces connection overhead by 50-70%. This is the **biggest win** for Supabase.

### 2. Enable Query Logging (Temporary)

Add to `app/Providers/AppServiceProvider.php`:

```php
public function boot()
{
    if (config('app.debug')) {
        \DB::listen(function ($query) {
            if ($query->time > 200) { // Log queries > 200ms
                \Log::warning('Slow Query', [
                    'sql' => $query->sql,
                    'time' => $query->time . 'ms',
                    'bindings' => $query->bindings,
                ]);
            }
        });
    }
}
```

This will help identify slow queries in `storage/logs/laravel.log`.

### 3. Increase Cache Times

For data that doesn't change often, increase cache duration:

**In controllers, change:**
- `300` (5 min) → `1800` (30 min) for static data
- `300` (5 min) → `3600` (1 hour) for very static data

**Example:**
```php
// Before
cache()->remember('admin.dashboard.stats', 300, ...)

// After (if data doesn't change often)
cache()->remember('admin.dashboard.stats', 1800, ...)
```

### 4. Add Database Query Timeout

Already added to `config/database.php` - this prevents hanging queries.

## 📊 Performance Monitoring

### Check Current Performance

Run this to see query counts:

```bash
php artisan tinker
```

Then:
```php
\DB::enableQueryLog();
// Visit a page
\DB::getQueryLog(); // See all queries
```

### Monitor Supabase Dashboard

1. Go to Supabase Dashboard
2. Check **Database** → **Performance**
3. Look for slow queries
4. Check connection pool usage

## 🎯 Expected Results

After these changes:
- **Connection Pooler**: 30-50% faster
- **Longer Cache**: 70-90% faster for cached pages
- **Query Optimization**: Already done (70-90% fewer queries)

**Overall**: 2-5x faster page loads

## ⚠️ Common Issues

### 1. "Connection timeout"
- **Fix**: Use port 6543 (pooler) instead of 5432
- **Fix**: Check Supabase project is active (not paused)

### 2. "Too many connections"
- **Fix**: Use connection pooler (port 6543)
- **Fix**: Reduce cache TTL to avoid too many connections

### 3. "Slow queries"
- **Fix**: Check logs for queries > 200ms
- **Fix**: Add indexes (already done)
- **Fix**: Use `select()` to limit columns (already done)

## 🔍 Debugging

### See All Queries on a Page

Add to any controller temporarily:

```php
\DB::enableQueryLog();
// ... your code ...
dd(\DB::getQueryLog()); // See all queries
```

### Check Cache Hit Rate

```php
// In tinker
cache()->get('admin.dashboard.stats'); // Should be fast if cached
```

## 📝 Next Steps

1. ✅ Switch to port 6543 (connection pooler)
2. ✅ Add query logging
3. ✅ Increase cache times for static data
4. ✅ Monitor slow queries
5. ⏭️ Consider Redis if still slow
6. ⏭️ Consider CDN for static assets

