# Supabase Performance Optimization Guide

## Is Supabase the Issue?

**Short Answer**: Supabase (PostgreSQL) can contribute to slowness, but it's usually **network latency** and **connection management** rather than Supabase itself.

### Common Supabase Performance Issues:

1. **Network Latency** - Supabase is cloud-hosted, so every query has network round-trip time
2. **Connection Pooling** - Not using connection pooling can cause connection overhead
3. **Too Many Queries** - N+1 queries are especially slow with network latency
4. **Missing Indexes** - Slower queries on Supabase than local DB
5. **Large Result Sets** - Fetching too much data over network

## Immediate Optimizations

### 1. Enable Connection Pooling (CRITICAL for Supabase)

Update your `.env` to use Supabase's connection pooler:

```env
# Use connection pooler (port 6543) instead of direct connection (port 5432)
DB_HOST=db.xxxxx.supabase.co
DB_PORT=6543  # Connection pooler port (faster, handles connections better)
# OR use direct connection:
# DB_PORT=5432  # Direct connection (slower but more compatible)
```

**Why**: Connection pooling reuses connections, reducing overhead. Port 6543 is Supabase's pooler.

### 2. Optimize Database Connection Settings

Add to `config/database.php` in the `pgsql` connection:

```php
'pgsql' => [
    'driver' => 'pgsql',
    // ... existing settings ...
    'options' => [
        PDO::ATTR_PERSISTENT => false, // Don't use persistent connections with pooler
        PDO::ATTR_EMULATE_PREPARES => false, // Use native prepared statements
    ],
    'pool' => [
        'min' => 2,
        'max' => 10,
    ],
],
```

### 3. Add Query Timeout

Prevent long-running queries from hanging:

```php
// In config/database.php
'pgsql' => [
    // ... existing ...
    'options' => [
        PDO::ATTR_TIMEOUT => 5, // 5 second timeout
    ],
],
```

### 4. Use Redis for Caching (Instead of File Cache)

Supabase queries are slow, so caching is critical. Use Redis:

```env
# In .env
CACHE_DRIVER=redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
```

**Install Redis**:
- Windows: Use WSL or Docker
- Or use Supabase's built-in Redis (if available)

### 5. Implement Database Query Logging

Add to `AppServiceProvider.php` to see slow queries:

```php
use Illuminate\Support\Facades\DB;

public function boot()
{
    if (config('app.debug')) {
        DB::listen(function ($query) {
            if ($query->time > 100) { // Log queries > 100ms
                \Log::warning('Slow query detected', [
                    'sql' => $query->sql,
                    'time' => $query->time,
                ]);
            }
        });
    }
}
```

## Code-Level Optimizations

### 1. Reduce Database Queries

Already done in previous optimizations, but ensure:
- ✅ Batch loading (done)
- ✅ Eager loading (done)
- ✅ Caching (done)
- ✅ Select specific columns (done)

### 2. Add More Aggressive Caching

Update cache TTLs in controllers:

```php
// Instead of 300 seconds (5 min), use longer for static data
cache()->remember('admin.dashboard.stats', 3600, function () { // 1 hour
    // ...
});
```

### 3. Use Database Transactions Wisely

Wrap multiple operations:

```php
DB::transaction(function () {
    // Multiple operations
});
```

### 4. Optimize Sidebar Queries

The sidebar makes queries on every page load. Cache them:

```php
// In sidebar component
$pendingCount = cache()->remember('trainer.pending_reviews.' . auth()->id(), 300, function() {
    // ... query ...
});
```

## Frontend Optimizations

### 1. Lazy Load Images

```html
<img loading="lazy" src="...">
```

### 2. Defer JavaScript

```html
<script defer src="..."></script>
```

### 3. Use CDN for Assets

If possible, serve static assets from CDN.

### 4. Minimize CSS/JS

Ensure Vite is minifying in production:

```bash
npm run build
```

## Supabase-Specific Optimizations

### 1. Use Supabase Connection Pooler

Always use port **6543** for connection pooling (already mentioned above).

### 2. Enable Supabase Database Indexes

Run this to check missing indexes:

```sql
-- Check for missing indexes
SELECT 
    schemaname,
    tablename,
    attname,
    n_distinct,
    correlation
FROM pg_stats
WHERE schemaname = 'public'
AND n_distinct > 100
AND correlation < 0.1
ORDER BY n_distinct DESC;
```

### 3. Use Supabase Edge Functions (Advanced)

For heavy operations, consider Supabase Edge Functions instead of Laravel.

### 4. Monitor Supabase Dashboard

Check Supabase dashboard for:
- Slow queries
- Connection pool usage
- Database size
- Query performance

## Quick Wins Checklist

- [ ] Switch to connection pooler (port 6543)
- [ ] Enable Redis caching
- [ ] Increase cache TTLs for static data
- [ ] Add query timeouts
- [ ] Monitor slow queries
- [ ] Use `select()` to limit columns
- [ ] Ensure all list views use pagination
- [ ] Cache sidebar counts
- [ ] Minimize CSS/JS in production

## Expected Improvements

After these optimizations:
- **Connection pooling**: 30-50% faster queries
- **Redis caching**: 70-90% faster for cached data
- **Query optimization**: 50-80% fewer queries
- **Overall**: 2-5x faster page loads

## Monitoring

Add this to monitor performance:

```php
// In AppServiceProvider
DB::listen(function ($query) {
    if ($query->time > 500) {
        \Log::error('Very slow query', [
            'sql' => $query->sql,
            'bindings' => $query->bindings,
            'time' => $query->time . 'ms',
        ]);
    }
});
```

## Next Steps

1. **Immediate**: Switch to connection pooler (port 6543)
2. **Short-term**: Enable Redis caching
3. **Medium-term**: Monitor and optimize slow queries
4. **Long-term**: Consider database read replicas if needed

