# Term Change Behavior - System Design

## Current System Behavior ✅

### When a Term Changes (New Term Activated):

1. **Classes Are NOT Deleted** ✅
   - Classes belong to a specific term (`term_id` in `classes` table)
   - When a term status changes from `active` → `completed`, classes are NOT deleted
   - Classes remain in the database for historical records and audit purposes
   - Old term classes become read-only (can't edit completed term data)

2. **Unit Allocations Persist with Classes** ✅
   - Unit allocations are stored in `class_unit` pivot table (many-to-many relationship)
   - Allocations are tied to classes, not directly to terms
   - Old term classes keep their unit allocations (historical data)
   - New term classes need NEW unit allocations to be created

3. **Trainer Allocations Persist with Classes** ✅
   - Trainer allocations are stored in `trainer_unit_class` pivot table
   - Allocations are tied to unit-class combinations, not directly to terms
   - Old term classes keep their trainer allocations (historical data)
   - New term classes need NEW trainer allocations to be created

## Data Flow When Term Changes:

### Old Term (Term 1 - Status: completed):
```
Term 1 (completed)
  ├── Class A (term_id = 1)
  │     ├── Unit X allocation (class_unit)
  │     └── Trainer T allocation (trainer_unit_class)
  └── Class B (term_id = 1)
        ├── Unit Y allocation (class_unit)
        └── Trainer U allocation (trainer_unit_class)
```

### New Term (Term 2 - Status: active):
```
Term 2 (active)
  ├── Class C (term_id = 2) ← NEW class created
  │     ├── (No unit allocations yet - need to create)
  │     └── (No trainer allocations yet - need to create)
  └── Class D (term_id = 2) ← NEW class created
        ├── (No unit allocations yet - need to create)
        └── (No trainer allocations yet - need to create)
```

**Old classes (A, B) remain in database - NOT deleted** ✅

## Filtering Behavior:

### Class Management:
- Default filter: Shows active term classes
- Can view all terms by selecting "All Terms" filter
- Old term classes are visible but read-only

### Unit Allocations:
- Default filter: Shows allocations for active term classes only
- Old term allocations remain in database but filtered out

### Trainer Allocations:
- Default filter: Shows allocations for active term classes only
- Old term allocations remain in database but filtered out

### Trainer "Marks Management":
- Shows units for active term classes only (recently fixed)
- Old term units are filtered out automatically

## Protection Against Data Loss:

1. **Term Deletion Protection**:
   ```php
   // Cannot delete term with associated classes
   if ($term->classes()->count() > 0) {
       return error('Cannot delete term with associated classes or units.');
   }
   ```

2. **Class Deletion Protection**:
   ```php
   // Cannot delete class with enrollments or submissions
   if ($class->enrollments()->count() > 0 || $class->poeSubmissions()->count() > 0) {
       return error('Cannot delete class with associated enrollments or submissions.');
   }
   ```

3. **Term Status Change**:
   - Only status changes (active → completed)
   - No data deletion occurs
   - All historical data preserved

## Summary:

✅ **Classes are NOT deleted when terms change**  
✅ **Unit allocations remain with their classes**  
✅ **Trainer allocations remain with their classes**  
✅ **New term requires NEW classes to be created**  
✅ **New term requires NEW allocations for new classes**  
✅ **Old term data is preserved for historical records**  

**System is working as designed!** 🎉
