# Term Management System Documentation

## Overview

The EPOE system operates using an **Active Term Model**, where system behavior automatically changes based on the currently active academic term. This ensures all academic and assessment data is properly scoped to terms while preserving historical data for audits and reporting.

## Key Features

### 1. Active Term Model
- **Only one term can be active at a time**
- When a new term is activated, the previous active term is automatically closed
- Closed terms become read-only for audit purposes
- All system operations are scoped to the active term

### 2. Term Statuses
- **Draft**: Term is created but not yet active
- **Active**: Currently operational term (only one allowed)
- **Completed**: Closed term, read-only
- **Archived**: Historical term, read-only

### 3. Term-Scoped Data
All academic data is tied to a term via `term_id`:
- Classes
- Units (via `term_department_units` table)
- Student enrollments (via classes)
- PoE submissions
- Marks and grading
- Validation records

## Implementation

### ActiveTermService

A service class provides easy access to the active term throughout the application:

```php
use App\Services\ActiveTermService;

// Get active term
$activeTerm = ActiveTermService::getActiveTerm();

// Get active term ID
$termId = ActiveTermService::getActiveTermId();

// Check if active term exists
if (ActiveTermService::hasActiveTerm()) {
    // Do something
}

// Scope query to active term
$classes = SchoolClass::query()
    ->where('term_id', ActiveTermService::getActiveTermId())
    ->get();

// Or use the helper
ActiveTermService::scopeToActiveTerm($query, 'term_id');
```

### Term Model Methods

```php
$term = Term::find(1);

// Check status
$term->isActive();      // true if status is 'active'
$term->isDraft();       // true if status is 'draft'
$term->isCompleted();   // true if status is 'completed'
$term->isClosed();       // true if completed or archived
$term->isReadOnly();    // true if closed

// Activate term (closes previous active term)
$term->activate();

// Close term
$term->close();

// Get duration
$days = $term->getDurationInDays();
```

### Scopes

```php
// Get active term
Term::active()->first();

// Get draft terms
Term::draft()->get();

// Get completed terms
Term::completed()->get();

// Get closed terms (completed or archived)
Term::closed()->get();
```

## Usage Examples

### Creating a Term

```php
// Create as draft
$term = Term::create([
    'name' => 'Term 1 - 2025',
    'start_date' => '2025-01-01',
    'end_date' => '2025-04-30',
    'status' => 'draft',
]);

// Or create and activate immediately
$term = Term::create([
    'name' => 'Term 1 - 2025',
    'start_date' => '2025-01-01',
    'end_date' => '2025-04-30',
    'status' => 'active', // Will close previous active term
]);
```

### Activating a Term

```php
// Via controller
POST /admin/terms/{term}/activate

// Programmatically
$term->activate();
ActiveTermService::clearCache();
```

### Querying Term-Scoped Data

```php
// Get classes for active term
$activeTermId = ActiveTermService::getActiveTermId();
$classes = SchoolClass::where('term_id', $activeTermId)->get();

// Get students enrolled in active term
$students = User::whereHas('enrollments.schoolClass', function($query) {
    $query->where('term_id', ActiveTermService::getActiveTermId());
})->get();

// Get PoE submissions for active term
$submissions = PoeSubmission::whereHas('schoolClass', function($query) {
    $query->where('term_id', ActiveTermService::getActiveTermId());
})->get();
```

## Admin Interface

### Term Management Routes

- `GET /admin/terms` - List all terms
- `GET /admin/terms/create` - Create new term form
- `POST /admin/terms` - Store new term
- `GET /admin/terms/{term}` - View term details
- `GET /admin/terms/{term}/edit` - Edit term form
- `PUT /admin/terms/{term}` - Update term
- `DELETE /admin/terms/{term}` - Delete term (draft only)
- `POST /admin/terms/{term}/activate` - Activate term
- `POST /admin/terms/{term}/close` - Close term

### Features

1. **Active Term Banner**: Shows current active term at top of list
2. **Status Indicators**: Color-coded status badges
3. **One-Click Activation**: Activate any draft term with one click
4. **Statistics**: View classes, units, students, and submissions per term
5. **Read-Only Protection**: Closed terms cannot be edited or deleted

## Data Integrity

### Protection Rules

1. **Active terms cannot be deleted**
2. **Closed terms cannot be edited or deleted** (audit requirement)
3. **Only draft terms can be deleted** (and only if no associated data)
4. **Activating a term automatically closes the previous active term**

### Historical Data Access

- All closed terms remain in the database
- Historical data is accessible for reporting
- Use `Term::closed()->get()` to access historical terms
- Reports can filter by specific term IDs

## Best Practices

1. **Always check for active term** before performing term-scoped operations
2. **Use ActiveTermService** instead of direct queries when possible
3. **Clear cache** after term activation/deactivation
4. **Validate term status** before allowing edits
5. **Preserve historical data** - never delete closed terms

## Future Enhancements

- [ ] Global query scopes for automatic term filtering
- [ ] Middleware to enforce active term requirement
- [ ] Term-based reporting dashboard
- [ ] Bulk operations for term data migration
- [ ] Term templates for quick setup
- [ ] Automatic term closure based on end date

## Related Models

- `Term` - Academic terms
- `SchoolClass` - Classes (belongs to Term)
- `TermDepartmentUnit` - Unit allocations (belongs to Term)
- `Enrollment` - Student enrollments (via Classes)
- `PoeSubmission` - Portfolio submissions (via Classes)

