# Troubleshooting 500 Internal Server Error

This guide will help you systematically diagnose and fix the 500 error on your TrueHost deployment.

## Step 1: Check Error Logs (Most Important!)

The error logs will tell you exactly what's wrong. Check these locations:

### Option A: Laravel Log File

1. **Via cPanel File Manager:**
   - Log in to cPanel
   - Go to **File Manager**
   - Navigate to `public_html/storage/logs/`
   - Open `laravel.log` (or the latest log file)
   - Scroll to the bottom to see the most recent errors

2. **Via FTP/SFTP:**
   - Connect to your server
   - Navigate to `storage/logs/`
   - Download `laravel.log`
   - Open it in a text editor and check the last 50-100 lines

### Option B: PHP Error Log

1. **Via cPanel:**
   - Go to **Error Log** section in cPanel
   - Check the latest errors

2. **Via File Manager:**
   - Look for `error_log` file in `public_html/` directory

### Option C: Apache Error Log

1. **Via cPanel:**
   - Go to **Metrics** → **Errors**
   - Check Apache error logs

**What to look for:**
- Missing `.env` file
- Missing `APP_KEY`
- Database connection errors
- Permission errors
- Missing PHP extensions
- Class not found errors

---

## Step 2: Enable Debug Mode (Temporarily)

If you can't access logs, temporarily enable debug mode to see the actual error:

1. **Via cPanel File Manager:**
   - Navigate to your application root
   - Open `.env` file
   - Change these lines:
     ```env
     APP_DEBUG=true
     APP_ENV=local
     ```
   - Save the file
   - Refresh your website

2. **Important:** After fixing the issue, change back to:
   ```env
   APP_DEBUG=false
   APP_ENV=production
   ```

---

## Step 3: Common Fixes (Try These in Order)

### Fix 1: Check if .env File Exists

**Problem:** Missing `.env` file

**Solution:**
1. Via cPanel File Manager, check if `.env` exists in your application root
2. If missing:
   - Copy `.env.example` to `.env` (if available)
   - Or create a new `.env` file with basic configuration:
     ```env
     APP_NAME="E-PoE System"
     APP_ENV=production
     APP_KEY=
     APP_DEBUG=false
     APP_URL=https://yourdomain.com
     
     DB_CONNECTION=pgsql
     DB_HOST=localhost
     DB_PORT=5432
     DB_DATABASE=your_database_name
     DB_USERNAME=your_username
     DB_PASSWORD=your_password
     ```

### Fix 2: Generate APP_KEY

**Problem:** Missing or invalid `APP_KEY`

**Solution:**

**If you have SSH access:**
```bash
cd /home/username/public_html
php artisan key:generate
```

**If you don't have SSH access:**
1. On your local machine, run:
   ```bash
   php artisan key:generate --show
   ```
2. Copy the generated key
3. In cPanel File Manager, open `.env` file
4. Add the key:
   ```env
   APP_KEY=base64:your_generated_key_here
   ```
5. Save the file

### Fix 3: Fix File Permissions

**Problem:** Incorrect file permissions

**Solution:**

**If you have SSH access:**
```bash
cd /home/username/public_html

# Set correct permissions
find . -type f -exec chmod 644 {} \;
find . -type d -exec chmod 755 {} \;

# Make storage and cache writable
chmod -R 775 storage bootstrap/cache
chown -R username:username storage bootstrap/cache
```

**If using cPanel File Manager:**
1. Navigate to `storage/` directory
2. Right-click → **Change Permissions**
3. Set to **775**
4. Check **Recurse into subdirectories**
5. Click **Change Permissions**
6. Repeat for `bootstrap/cache/` directory

### Fix 4: Check Database Connection

**Problem:** Database connection failing

**Solution:**
1. Verify database credentials in `.env`:
   ```env
   DB_CONNECTION=pgsql
   DB_HOST=localhost  # Or the host provided by TrueHost
   DB_PORT=5432
   DB_DATABASE=your_full_database_name
   DB_USERNAME=your_full_username
   DB_PASSWORD=your_password
   ```

2. **Important:** TrueHost may use a different host. Check:
   - cPanel → **PostgreSQL Databases**
   - Look for connection details or hostname
   - It might be `localhost`, `127.0.0.1`, or a specific hostname

3. Test connection via cPanel PostgreSQL tools

### Fix 5: Ensure Vendor Directory Exists

**Problem:** Missing `vendor/` directory (Composer dependencies)

**Solution:**

**If you have SSH access:**
```bash
cd /home/username/public_html
composer install --no-dev --optimize-autoloader
```

**If you don't have SSH access:**
1. On your local machine:
   ```bash
   composer install --no-dev --optimize-autoloader
   ```
2. Upload the `vendor/` directory to your server via FTP

### Fix 6: Create Storage Directories

**Problem:** Missing storage directories

**Solution:**

**If you have SSH access:**
```bash
cd /home/username/public_html
mkdir -p storage/app/public/poe/assignments
mkdir -p storage/app/public/poe/practical-evidence
mkdir -p storage/app/public/poe/submissions
mkdir -p storage/app/backups
mkdir -p storage/framework/cache
mkdir -p storage/framework/sessions
mkdir -p storage/framework/views
mkdir -p storage/logs
chmod -R 775 storage
```

**If using cPanel File Manager:**
1. Navigate to `storage/` directory
2. Create these directories if they don't exist:
   - `app/public/poe/assignments`
   - `app/public/poe/practical-evidence`
   - `app/public/poe/submissions`
   - `app/backups`
   - `framework/cache`
   - `framework/sessions`
   - `framework/views`
   - `logs`
3. Set permissions to 775 for all storage directories

### Fix 7: Check PHP Version and Extensions

**Problem:** Wrong PHP version or missing extensions

**Solution:**
1. In cPanel, go to **Select PHP Version**
2. Ensure PHP 8.0 or higher is selected (recommended: 8.1 or 8.2)
3. Click **Extensions** and enable:
   - `pdo_pgsql`
   - `pgsql`
   - `mbstring`
   - `openssl`
   - `tokenizer`
   - `xml`
   - `ctype`
   - `json`
   - `fileinfo`
   - `gd`
   - `zip`
   - `curl`

### Fix 8: Clear All Caches

**Problem:** Cached configuration causing issues

**Solution:**

**If you have SSH access:**
```bash
cd /home/username/public_html
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

**If you don't have SSH access:**
1. Delete these directories via File Manager:
   - `bootstrap/cache/*` (delete contents, not the directory)
   - `storage/framework/cache/*`
   - `storage/framework/views/*`
2. Create empty `.gitkeep` files in these directories if needed

### Fix 9: Check .htaccess File

**Problem:** Missing or incorrect `.htaccess` file

**Solution:**
1. Ensure `.htaccess` exists in `public/` directory (or `public_html/` if that's your document root)
2. Content should be:
   ```apache
   <IfModule mod_rewrite.c>
       <IfModule mod_negotiation.c>
           Options -MultiViews -Indexes
       </IfModule>

       RewriteEngine On

       # Handle Authorization Header
       RewriteCond %{HTTP:Authorization} .
       RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

       # Redirect Trailing Slashes If Not A Folder...
       RewriteCond %{REQUEST_FILENAME} !-d
       RewriteCond %{REQUEST_URI} (.+)/$
       RewriteRule ^ %1 [L,R=301]

       # Send Requests To Front Controller...
       RewriteCond %{REQUEST_FILENAME} !-d
       RewriteCond %{REQUEST_FILENAME} !-f
       RewriteRule ^ index.php [L]
   </IfModule>
   ```

3. Ensure `mod_rewrite` is enabled in cPanel

### Fix 10: Verify Document Root

**Problem:** Document root not pointing to correct directory

**Solution:**
1. In cPanel, check your domain's document root
2. It should point to `public_html/public/` (if Laravel structure) or `public_html/` (if you moved files)
3. If using subdirectory structure, ensure `public/index.php` paths are correct

---

## Step 4: Quick Diagnostic Checklist

Run through this checklist:

- [ ] `.env` file exists in application root
- [ ] `APP_KEY` is set in `.env`
- [ ] `vendor/` directory exists and is uploaded
- [ ] `storage/` directory has 775 permissions
- [ ] `bootstrap/cache/` directory has 775 permissions
- [ ] All storage subdirectories exist
- [ ] Database credentials in `.env` are correct
- [ ] PHP version is 8.0 or higher
- [ ] Required PHP extensions are enabled
- [ ] `.htaccess` file exists in `public/` directory
- [ ] `mod_rewrite` is enabled
- [ ] Document root is correct

---

## Step 5: Test After Each Fix

After applying each fix:
1. Clear browser cache (Ctrl+F5 or Cmd+Shift+R)
2. Try accessing your website
3. Check error logs again if still getting 500 error
4. Move to next fix if needed

---

## Step 6: Common Error Messages and Solutions

### "No application encryption key has been specified"
**Solution:** Generate `APP_KEY` (see Fix 2)

### "SQLSTATE[08006] [7] could not connect to server"
**Solution:** Check database credentials and host (see Fix 4)

### "The stream or file could not be opened: permission denied"
**Solution:** Fix storage permissions (see Fix 3)

### "Class 'X' not found"
**Solution:** Run `composer install` or upload `vendor/` directory (see Fix 5)

### "Route [X] not defined"
**Solution:** Clear route cache: `php artisan route:clear`

### "Vite manifest not found"
**Solution:** Build and upload Vite assets (see deployment guide)

---

## Step 7: Still Not Working?

If none of the above fixes work:

1. **Check Laravel log file** (`storage/logs/laravel.log`) for the exact error
2. **Enable debug mode** temporarily to see the error message
3. **Contact TrueHost support** with:
   - Your error log contents
   - PHP version
   - Enabled PHP extensions
   - Database connection details (without password)
4. **Verify your deployment** matches the deployment guide exactly

---

## Quick Fix Script (If You Have SSH Access)

If you have SSH access, run this script to fix common issues:

```bash
#!/bin/bash
cd /home/username/public_html

# Fix permissions
find . -type f -exec chmod 644 {} \;
find . -type d -exec chmod 755 {} \;
chmod -R 775 storage bootstrap/cache

# Create storage directories
mkdir -p storage/app/public/poe/{assignments,practical-evidence,submissions}
mkdir -p storage/app/backups
mkdir -p storage/framework/{cache,sessions,views}
mkdir -p storage/logs

# Clear caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Generate key if missing
if ! grep -q "APP_KEY=base64:" .env; then
    php artisan key:generate
fi

echo "Common fixes applied. Check your website now."
```

---

## Prevention Tips

To avoid 500 errors in the future:

1. **Always test locally** before deploying
2. **Follow the deployment guide** step by step
3. **Keep error logs enabled** in production
4. **Regularly backup** your `.env` file
5. **Document your configuration** for easy recovery

---

## Need More Help?

If you're still stuck:
1. Share the error message from `storage/logs/laravel.log`
2. Share your `.env` file structure (without sensitive data)
3. Share PHP version and enabled extensions
4. Check TrueHost documentation for specific requirements

---

**Last Updated:** January 2026
