<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\SchoolClass;
use App\Services\ActiveTermService;
use App\Services\QueryOptimizer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ApprovalController extends Controller
{
    /**
     * Display a listing of submissions pending approval.
     */
    public function index(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || !$hod->department_id) {
            return view('hod.approvals.index', [
                'submissions' => collect(),
                'activeTerm' => $activeTerm,
                'stats' => [],
            ])->with('warning', 'No active term or department assigned.');
        }

        // Get department classes
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        if (empty($departmentClassIds)) {
            return view('hod.approvals.index', [
                'submissions' => collect(),
                'activeTerm' => $activeTerm,
                'stats' => [],
            ])->with('info', 'No classes found for your department.');
        }

        // Build query for department submissions
        $query = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('class_id', $departmentClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            });

        // Apply filters
        if ($request->status) {
            if ($request->status === 'pending') {
                $query->where('status', 'trainer_reviewed');
            } else {
                $query->where('status', $request->status);
            }
        } else {
            // Default: show pending approvals
            $query->where('status', 'trainer_reviewed');
        }

        if ($request->class) {
            $query->where('class_id', $request->class);
        }

        if ($request->unit) {
            $query->where('unit_id', $request->unit);
        }

        if ($request->search) {
            $query->whereHas('student', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        $submissions = $query->with([
            'student:id,name,email',
            'unit:id,name,code',
            'schoolClass:id,name,code',
        ])->latest('submitted_at')->paginate(20);

        // Calculate statistics
        $stats = cache()->remember('hod.approvals.stats.' . $hod->id . '.' . $activeTerm->id, 300, function() use ($departmentClassIds, $activeTerm) {
            $baseQuery = PoeSubmission::withoutGlobalScope('activeTerm')
                ->whereIn('class_id', $departmentClassIds)
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                });
            
            return [
                'pending' => (clone $baseQuery)->where('status', 'trainer_reviewed')->count(),
                'approved' => (clone $baseQuery)->where('status', 'hod_approved')->count(),
                'rejected' => (clone $baseQuery)->where('status', 'rejected')->count(),
                'total' => (clone $baseQuery)->count(),
            ];
        });

        // Get filter options
        $filterOptions = [
            'classes' => SchoolClass::whereIn('id', $departmentClassIds)
                ->select('id', 'name', 'code')
                ->orderBy('name')
                ->get(),
            'units' => \App\Models\Unit::where('department_id', $hod->department_id)
                ->select('id', 'name', 'code')
                ->orderBy('name')
                ->get(),
        ];

        return view('hod.approvals.index', compact(
            'submissions',
            'activeTerm',
            'stats',
            'filterOptions'
        ));
    }

    /**
     * Display the specified submission.
     */
    public function show(PoeSubmission $submission)
    {
        $hod = Auth::user();
        
        // Verify submission belongs to HOD's department
        if (!$submission->schoolClass || $submission->schoolClass->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this submission.');
        }

        // Load submission details
        $submission->load([
            'student:id,name,email,phone,department_id' => ['department:id,name'],
            'unit:id,name,code,department_id' => ['department:id,name'],
            'schoolClass:id,name,code,term_id,department_id,level_id' => [
                'term:id,name,status',
                'department:id,name',
                'level:id,name'
            ],
            'evidence:id,poe_submission_id,file_name,file_type,file_size,created_at',
            'validation:id,poe_submission_id,validator_id,status,comments,validated_at' => [
                'validator:id,name,email'
            ]
        ]);

        return view('hod.approvals.show', compact('submission'));
    }

    /**
     * Approve a submission.
     */
    public function approve(Request $request, PoeSubmission $submission)
    {
        $hod = Auth::user();
        
        // Verify submission belongs to HOD's department
        if (!$submission->schoolClass || $submission->schoolClass->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to approve this submission.');
        }

        // Verify submission is in correct status
        if ($submission->status !== 'trainer_reviewed') {
            return redirect()->back()
                ->with('error', 'This submission is not pending approval.');
        }

        $request->validate([
            'comments' => ['nullable', 'string', 'max:1000'],
        ]);

        // Update submission status
        $submission->update([
            'status' => 'hod_approved',
        ]);

        // Create validation record (HOD approval)
        \App\Models\Validation::create([
            'poe_submission_id' => $submission->id,
            'validator_id' => $hod->id,
            'status' => 'approved',
            'comments' => $request->comments,
            'validated_at' => now(),
        ]);

        // Log audit
        \App\Models\AuditLog::create([
            'user_id' => $hod->id,
            'action' => 'approved_submission',
            'model_type' => PoeSubmission::class,
            'model_id' => $submission->id,
            'changes' => [
                'status' => ['trainer_reviewed', 'hod_approved'],
            ],
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('hod.approvals.index')
            ->with('success', 'Submission approved successfully.');
    }

    /**
     * Reject a submission.
     */
    public function reject(Request $request, PoeSubmission $submission)
    {
        $hod = Auth::user();
        
        // Verify submission belongs to HOD's department
        if (!$submission->schoolClass || $submission->schoolClass->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to reject this submission.');
        }

        // Verify submission is in correct status
        if ($submission->status !== 'trainer_reviewed') {
            return redirect()->back()
                ->with('error', 'This submission is not pending approval.');
        }

        $request->validate([
            'comments' => ['required', 'string', 'max:1000'],
        ]);

        // Update submission status
        $submission->update([
            'status' => 'rejected',
        ]);

        // Create validation record (HOD rejection)
        \App\Models\Validation::create([
            'poe_submission_id' => $submission->id,
            'validator_id' => $hod->id,
            'status' => 'rejected',
            'comments' => $request->comments,
            'validated_at' => now(),
        ]);

        // Log audit
        \App\Models\AuditLog::create([
            'user_id' => $hod->id,
            'action' => 'rejected_submission',
            'model_type' => PoeSubmission::class,
            'model_id' => $submission->id,
            'changes' => [
                'status' => ['trainer_reviewed', 'rejected'],
            ],
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('hod.approvals.index')
            ->with('success', 'Submission rejected.');
    }
}
