<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\PoeSubmission;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Models\ValidationRequest;
use App\Services\ActiveTermService;
use App\Services\RedisCacheService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class MarksController extends Controller
{
    /**
     * Display marks management page for department.
     */
    public function index(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('hod.marks.index', [
                'units' => collect(),
                'activeTerm' => null,
                'stats' => [
                    'total_units' => 0,
                    'total_assignments' => 0,
                    'total_students' => 0,
                    'average_marks' => 0,
                    'highest_marks' => 0,
                    'lowest_marks' => 0,
                ],
                'filterOptions' => [
                    'units' => collect(),
                    'classes' => collect(),
                ],
            ])->with('warning', 'No active term found.');
        }

        if (!$hod->department_id) {
            return view('hod.marks.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
                'stats' => [
                    'total_units' => 0,
                    'total_assignments' => 0,
                    'total_students' => 0,
                    'average_marks' => 0,
                    'highest_marks' => 0,
                    'lowest_marks' => 0,
                ],
                'filterOptions' => [
                    'units' => collect(),
                    'classes' => collect(),
                ],
            ])->with('warning', 'You are not assigned to a department.');
        }

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        if (empty($departmentClassIds)) {
            return view('hod.marks.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
                'stats' => [
                    'total_units' => 0,
                    'total_assignments' => 0,
                    'total_students' => 0,
                    'average_marks' => 0,
                    'highest_marks' => 0,
                    'lowest_marks' => 0,
                ],
                'filterOptions' => [
                    'units' => Unit::where('department_id', $hod->department_id)
                        ->select('id', 'name', 'code')
                        ->orderBy('name')
                        ->get(),
                    'classes' => collect(),
                ],
            ])->with('info', 'No classes found for your department.');
        }

        // Get all units in the department - with Redis caching
        $cacheKey = 'hod:marks:units:' . $hod->department_id . ':term:' . $activeTerm->id;
        $units = Cache::remember($cacheKey, RedisCacheService::TTL_MEDIUM, function() use ($hod, $request) {
            $unitsQuery = Unit::where('department_id', $hod->department_id)
                ->with(['department:id,name,code']);

            // Filter by search
            if ($request->has('search') && $request->search) {
                $search = $request->search;
                $unitsQuery->where(function($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('code', 'like', "%{$search}%");
                });
            }

            // Filter by unit
            if ($request->has('unit') && $request->unit) {
                $unitsQuery->where('id', $request->unit);
            }

            return $unitsQuery->orderBy('name')->get();
        });

        // Calculate statistics for each unit - with Redis caching
        $allMarks = [];
        $totalAssignments = 0;
        $totalStudents = 0;

        // Calculate unit statistics with Redis caching
        foreach ($units as $unit) {
            $unitCacheKey = 'hod:marks:unit:' . $unit->id . ':term:' . $activeTerm->id;
            $unitStats = Cache::remember($unitCacheKey, RedisCacheService::TTL_MEDIUM, function() use ($unit, $departmentClassIds, $activeTerm) {
                // Get classes that have this unit assigned
                $classIds = DB::table('class_unit')
                    ->where('unit_id', $unit->id)
                    ->whereIn('class_id', $departmentClassIds)
                    ->pluck('class_id')
                    ->toArray();

                // Get assignments for this unit (all trainers)
                $assignments = Assignment::where('unit_id', $unit->id)
                    ->where('is_published', true)
                    ->get();

                $assignmentsCount = $assignments->count();

                // Get students enrolled in classes with this unit
                $students = \App\Models\User::where('role', 'student')
                    ->whereHas('enrollments', function($q) use ($classIds, $activeTerm) {
                        $q->whereIn('class_id', $classIds)
                          ->where('status', 'active');
                        if ($activeTerm) {
                            $q->whereHas('schoolClass', function($qc) use ($activeTerm) {
                                $qc->where('term_id', $activeTerm->id);
                            });
                        }
                    })
                    ->count();

                // Get all marks for this unit
                $unitMarks = [];
                foreach ($assignments as $assignment) {
                    $submissions = AssignmentSubmission::where('assignment_id', $assignment->id)
                        ->whereNotNull('marks')
                        ->get();

                    foreach ($submissions as $submission) {
                        $unitMarks[] = $submission->marks;
                    }
                }

                // Calculate unit statistics
                $stats = [
                    'assignments_count' => $assignmentsCount,
                    'students_count' => $students,
                    'average_marks' => count($unitMarks) > 0 ? round(array_sum($unitMarks) / count($unitMarks), 2) : 0,
                    'highest_marks' => count($unitMarks) > 0 ? max($unitMarks) : 0,
                    'lowest_marks' => count($unitMarks) > 0 ? min($unitMarks) : 0,
                    'total_marks_count' => count($unitMarks),
                    'unit_marks' => $unitMarks, // Store for overall stats calculation
                ];

                return $stats;
            });

            // Assign cached stats to unit object
            $unit->assignments_count = $unitStats['assignments_count'];
            $unit->students_count = $unitStats['students_count'];
            $unit->average_marks = $unitStats['average_marks'];
            $unit->highest_marks = $unitStats['highest_marks'];
            $unit->lowest_marks = $unitStats['lowest_marks'];
            $unit->total_marks_count = $unitStats['total_marks_count'];

            // Collect marks for overall stats
            if (isset($unitStats['unit_marks'])) {
                $allMarks = array_merge($allMarks, $unitStats['unit_marks']);
            }
            $totalAssignments += $unit->assignments_count;
            $totalStudents += $unit->students_count;
        }

        // Per-unit validation contexts: classes (in department) that have this unit, with pending request if any
        foreach ($units as $unit) {
            $classIds = DB::table('class_unit')
                ->where('unit_id', $unit->id)
                ->whereIn('class_id', $departmentClassIds)
                ->pluck('class_id');
            $unit->validation_contexts = collect();
            if ($classIds->isNotEmpty()) {
                $classes = SchoolClass::whereIn('id', $classIds)->orderBy('name')->get();
                foreach ($classes as $class) {
                    $vr = ValidationRequest::where('class_id', $class->id)
                        ->where('unit_id', $unit->id)
                        ->where('term_id', $activeTerm->id)
                        ->where('status', 'pending')
                        ->first();
                    $unit->validation_contexts->push(['class' => $class, 'validation_request' => $vr]);
                }
            }
        }

        // Calculate overall statistics
        $stats = [
            'total_units' => $units->count(),
            'total_assignments' => $totalAssignments,
            'total_students' => $totalStudents,
            'average_marks' => count($allMarks) > 0 ? round(array_sum($allMarks) / count($allMarks), 2) : 0,
            'highest_marks' => count($allMarks) > 0 ? max($allMarks) : 0,
            'lowest_marks' => count($allMarks) > 0 ? min($allMarks) : 0,
        ];

        $filterOptions = [
            'units' => Unit::where('department_id', $hod->department_id)
                ->select('id', 'name', 'code')
                ->orderBy('name')
                ->get(),
            'classes' => SchoolClass::whereIn('id', $departmentClassIds)
                ->select('id', 'name', 'code')
                ->orderBy('name')
                ->get(),
        ];

        return view('hod.marks.index', compact(
            'units',
            'activeTerm',
            'filterOptions',
            'stats'
        ));
    }

    /**
     * Get validation readiness for a class + unit (HOD: any unit in department).
     */
    public function getValidationReadinessForHod(SchoolClass $class, Unit $unit): array
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $class->term_id !== $activeTerm->id) {
            return ['ready' => false, 'errors' => ['Invalid or inactive term.']];
        }

        if ($unit->department_id !== $hod->department_id || $class->department_id !== $hod->department_id) {
            return ['ready' => false, 'errors' => ['Unit or class does not belong to your department.']];
        }

        if (!$unit->isConfigured()) {
            return [
                'ready' => false,
                'errors' => ['Unit assessment structure has not been configured.'],
                'unit_not_configured' => true,
            ];
        }

        $studentIds = $class->students()
            ->wherePivot('status', 'active')
            ->pluck('users.id');

        if ($studentIds->isEmpty()) {
            return ['ready' => false, 'errors' => ['No active students found in this class.']];
        }

        $expectedCount = $studentIds->count();
        $errors = [];
        $structure = $unit->getAssessmentStructure();

        foreach (['theory', 'practical', 'oral', 'project'] as $type) {
            $required = $structure[$type] ?? 0;
            if ($required > 0) {
                $assignmentTypeMap = [
                    'theory' => ['assignment', 'exam'],
                    'practical' => ['practical'],
                    'oral' => ['practical'],
                    'project' => ['project'],
                ];
                $typesToCheck = $assignmentTypeMap[$type] ?? [$type];
                $completedCount = Assignment::where('unit_id', $unit->id)
                    ->where('class_id', $class->id)
                    ->where('is_published', true)
                    ->whereIn('type', $typesToCheck)
                    ->get()
                    ->filter(function ($assignment) use ($studentIds, $expectedCount) {
                        $markedCount = AssignmentSubmission::where('assignment_id', $assignment->id)
                            ->whereIn('student_id', $studentIds)
                            ->whereIn('marking_status', ['marked', 'absent'])
                            ->count();
                        return $markedCount >= $expectedCount;
                    })
                    ->count();

                if ($completedCount < $required) {
                    $missing = $required - $completedCount;
                    $typeLabel = ucfirst($type);
                    $errors[] = "Missing: {$missing} {$typeLabel} Assessment(s) (Required: {$required}, Completed: {$completedCount})";
                }
            }
        }

        $allAssignments = Assignment::where('unit_id', $unit->id)
            ->where('class_id', $class->id)
            ->where('is_published', true)
            ->get();

        foreach ($allAssignments as $assignment) {
            $markedCount = AssignmentSubmission::where('assignment_id', $assignment->id)
                ->whereIn('student_id', $studentIds)
                ->whereIn('marking_status', ['marked', 'absent'])
                ->count();
            if ($markedCount < $expectedCount) {
                $missing = $expectedCount - $markedCount;
                $errors[] = "{$assignment->title} ({$assignment->type}): {$missing} student(s) missing marks.";
            }
        }

        return [
            'ready' => empty($errors),
            'errors' => $errors,
            'unit_classification' => $unit->classification,
            'unit_name' => $unit->name,
        ];
    }

    /**
     * Push a unit (class + unit) to the validation portal. HOD can push any unit in the department.
     */
    public function pushToValidation(Request $request, SchoolClass $class, Unit $unit)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || $class->term_id !== $activeTerm->id) {
            return back()->with('validation_push_error', 'Cannot push to validation: invalid or inactive term.');
        }

        if ($unit->department_id !== $hod->department_id || $class->department_id !== $hod->department_id) {
            abort(403, 'You can only push units and classes in your department.');
        }

        if (ValidationRequest::isLocked($class->id, $unit->id, $activeTerm->id)) {
            return back()->with('validation_push_error', 'This class is already under validation or approved for this unit.');
        }

        $readiness = $this->getValidationReadinessForHod($class, $unit);
        if (!$readiness['ready']) {
            return back()
                ->with('validation_push_error', 'Cannot push to validation. ' . implode(' ', $readiness['errors'] ?? []))
                ->with('validation_errors', $readiness['errors'] ?? []);
        }

        ValidationRequest::create([
            'class_id' => $class->id,
            'unit_id' => $unit->id,
            'term_id' => $activeTerm->id,
            'submitted_by' => $hod->id,
            'submitted_role' => 'hod',
            'status' => 'pending',
            'submitted_at' => now(),
        ]);

        return back()->with('success', "Unit \"{$unit->name}\" for class \"{$class->name}\" has been pushed to the validation portal.");
    }

    /**
     * Cancel a pending push request (remove from validation portal).
     */
    public function cancelValidationRequest(Request $request, ValidationRequest $validationRequest)
    {
        $hod = Auth::user();

        if ($validationRequest->status !== 'pending') {
            return back()->with('error', 'Only pending validation requests can be cancelled.');
        }

        $class = $validationRequest->schoolClass;
        $unit = $validationRequest->unit;
        if (!$class || !$unit || $class->department_id !== $hod->department_id || $unit->department_id !== $hod->department_id) {
            abort(403, 'You can only cancel validation requests for your department.');
        }

        $validationRequest->delete();
        return back()->with('success', 'Validation request has been cancelled. The unit is no longer in the validation portal.');
    }

    /**
     * Print unit marks summary for HOD
     */
    public function printUnitMarks(Request $request, $unitId)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        $unit = Unit::with('department:id,name')->findOrFail($unitId);
        
        // Verify unit belongs to HOD's department
        if ($unit->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this unit.');
        }
        
        // Get unit configuration
        $assessmentStructure = $unit->getAssessmentStructure();
        
        // Map assignment "type" values to assessment categories used in configuration/printing
        // - "assignment" + "exam"  => "theory"
        // - "practical"            => "practical"
        // - "project"              => "project"
        // There is no separate "oral" assignment type; oral is handled as part of practical via assessor tools.
        $typeToCategoryMap = [
            'assignment' => 'theory',
            'exam'       => 'theory',
            'practical'  => 'practical',
            'project'    => 'project',
        ];

        // Get all assignments for this unit (all trainers in the department)
        $assignmentsQuery = Assignment::where('unit_id', $unitId)
            ->where('is_published', true);
        
        // Filter by category if provided
        if ($request->has('category') && $request->category) {
            $assignmentsQuery->where('type', $request->category);
        }
        
        $allAssignments = $assignmentsQuery->orderBy('created_at', 'asc')->get();
        
        // Group assignments by assessment category and number them
        $groupedAssignments = [
            'theory' => [],
            'practical' => [],
            'oral' => [],
            'project' => [],
        ];
        // Track raw type presence counts (for potential use in views / debugging)
        $typeCounts = [
            'assignment' => 0,
            'exam' => 0,
            'practical' => 0,
            'project' => 0,
        ];
        
        foreach ($allAssignments as $assignment) {
            if (isset($typeCounts[$assignment->type])) {
                $typeCounts[$assignment->type]++;
            }

            $category = $typeToCategoryMap[$assignment->type] ?? null;
            if ($category && isset($groupedAssignments[$category])) {
                $groupedAssignments[$category][] = $assignment;
            }
        }
        
        // Create a flat list with numbered assignments for display
        $assignments = [];
        $assignmentMap = [];
        
        foreach (['theory', 'practical', 'oral', 'project'] as $type) {
            $count = count($groupedAssignments[$type]);
            $requiredCount = $assessmentStructure[$type] ?? 0;
            
            if ($count > 0 || $requiredCount > 0) {
                foreach ($groupedAssignments[$type] as $index => $assignment) {
                    $number = $index + 1;
                    $prefix = $type === 'theory' ? 'EXAM' : strtoupper($type);
                    $assignment->display_name = $prefix . $number;
                    $assignment->display_type = $type;
                    $assignments[] = $assignment;
                    $assignmentMap[$assignment->id] = [
                        'display_name' => $assignment->display_name,
                        'type' => $type,
                        'number' => $number,
                    ];
                }
            }
        }

        // Get all students in department classes with this unit
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        $classIds = DB::table('class_unit')
            ->where('unit_id', $unitId)
            ->whereIn('class_id', $departmentClassIds)
            ->pluck('class_id')
            ->toArray();

        $students = \App\Models\User::where('role', 'student')
            ->whereHas('enrollments', function($q) use ($classIds, $activeTerm) {
                $q->whereIn('class_id', $classIds)
                  ->where('status', 'active');
                if ($activeTerm) {
                    $q->whereHas('schoolClass', function($qc) use ($activeTerm) {
                        $qc->where('term_id', $activeTerm->id);
                    });
                }
            })
            ->select('id', 'name', 'email', 'admission_number')
            ->orderBy('name')
            ->get();

        // Get marks for each student and assignment, and calculate category averages
        $marksData = [];
        $categoryAverages = [];
        
        foreach ($students as $student) {
            $studentMarks = [];
            $categoryTotals = [
                'theory' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'practical' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'oral' => ['total' => 0, 'count' => 0, 'max_total' => 0],
                'project' => ['total' => 0, 'count' => 0, 'max_total' => 0],
            ];
            
            foreach ($allAssignments as $assignment) {
                $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                    ->where('student_id', $student->id)
                    ->first();
                
                $marks = $submission ? $submission->marks : null;
                $percentage = null;
                
                if ($marks !== null && $assignment->total_marks > 0) {
                    $percentage = ($marks / $assignment->total_marks) * 100;
                    
                    // Add to category totals using mapped category key
                    $categoryKey = $typeToCategoryMap[$assignment->type] ?? null;
                    if ($categoryKey && isset($categoryTotals[$categoryKey])) {
                        $categoryTotals[$categoryKey]['total'] += $marks;
                        $categoryTotals[$categoryKey]['max_total'] += $assignment->total_marks;
                        $categoryTotals[$categoryKey]['count']++;
                    }
                }
                
                $studentMarks[$assignment->id] = [
                    'marks' => $marks,
                    'percentage' => $percentage,
                    'status' => $submission ? $submission->marking_status : 'not_marked',
                    'submission' => $submission,
                    'total_marks' => $assignment->total_marks,
                ];
            }
            
            // Calculate category averages in percentage
            $categoryAverages[$student->id] = [];
            foreach (['theory', 'practical', 'oral', 'project'] as $type) {
                $cat = $categoryTotals[$type];
                if ($cat['count'] > 0 && $cat['max_total'] > 0) {
                    $categoryAverages[$student->id][$type] = [
                        'average_percentage' => ($cat['total'] / $cat['max_total']) * 100,
                        'count' => $cat['count'],
                        'total_marks' => $cat['total'],
                        'max_total' => $cat['max_total'],
                    ];
                } else {
                    $categoryAverages[$student->id][$type] = [
                        'average_percentage' => null,
                        'count' => 0,
                        'total_marks' => 0,
                        'max_total' => 0,
                    ];
                }
            }
            
            $marksData[$student->id] = $studentMarks;
        }

        // Get institution settings for letterhead
        $institutionSettings = \App\Models\Setting::where('key', 'like', 'institution.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });

        // Track GA4 event for report generation
        $ga4Event = [
            'name' => 'report_generated',
            'params' => [
                'report_type' => 'unit_marks',
                'user_role' => 'hod',
            ],
        ];

        return view('hod.marks.print-unit', compact(
            'unit', 
            'assignments', 
            'students', 
            'marksData', 
            'activeTerm',
            'groupedAssignments',
            'categoryAverages',
            'assessmentStructure',
            'institutionSettings',
            'hod',
            'ga4Event'
        ));
    }

    /**
     * Print consolidated class marks - all units for a class in one sheet
     */
    public function printClassMarks(Request $request, $classId)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        $class = SchoolClass::with(['department:id,name', 'term:id,name', 'level:id,name'])->findOrFail($classId);
        
        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this class.');
        }
        
        // Get all units assigned to this class
        $units = $class->units()
            ->orderBy('name')
            ->get();
        
        if ($units->isEmpty()) {
            return back()->with('error', 'This class has no units assigned.');
        }
        
        // Get all students enrolled in this class
        $students = \App\Models\User::where('role', 'student')
            ->whereHas('enrollments', function($q) use ($class, $activeTerm) {
                $q->where('class_id', $class->id)
                  ->where('status', 'active');
                if ($activeTerm) {
                    $q->whereHas('schoolClass', function($qc) use ($activeTerm) {
                        $qc->where('term_id', $activeTerm->id);
                    });
                }
            })
            ->select('id', 'name', 'email', 'admission_number')
            ->orderBy('name')
            ->get();
        
        // For each student, calculate average marks for each unit
        $studentMarks = [];
        
        foreach ($students as $student) {
            $marksRow = [
                'student' => $student,
                'units' => [],
            ];
            
            foreach ($units as $unit) {
                // Get all published assignments for this unit
                $assignments = Assignment::where('unit_id', $unit->id)
                    ->where('is_published', true)
                    ->get();
                
                if ($assignments->isEmpty()) {
                    $marksRow['units'][$unit->id] = [
                        'average' => null,
                        'total_marks' => 0,
                        'max_total' => 0,
                    ];
                    continue;
                }
                
                // Calculate total marks for this student across all assignments in this unit
                $totalMarks = 0;
                $maxTotal = 0;
                $markedAssignmentsCount = 0;
                
                foreach ($assignments as $assignment) {
                    $submission = AssignmentSubmission::where('assignment_id', $assignment->id)
                        ->where('student_id', $student->id)
                        ->first();
                    
                    if ($submission && $submission->marks !== null) {
                        $totalMarks += $submission->marks;
                        $maxTotal += $assignment->total_marks;
                        $markedAssignmentsCount++;
                    } else {
                        $maxTotal += $assignment->total_marks;
                    }
                }
                
                // Calculate average (raw marks average across marked assignments)
                if ($markedAssignmentsCount > 0) {
                    $average = $totalMarks / $markedAssignmentsCount;
                    $marksRow['units'][$unit->id] = [
                        'average' => $average,
                        'total_marks' => $totalMarks,
                        'max_total' => $maxTotal,
                        'assignment_count' => $markedAssignmentsCount,
                    ];
                } else {
                    $marksRow['units'][$unit->id] = [
                        'average' => null,
                        'total_marks' => 0,
                        'max_total' => $maxTotal,
                        'assignment_count' => 0,
                    ];
                }
            }
            
            $studentMarks[] = $marksRow;
        }
        
        // Get institution settings for letterhead
        $institutionSettings = \App\Models\Setting::where('key', 'like', 'institution.%')
            ->get()
            ->keyBy('key')
            ->map(function($setting) {
                return $setting->value;
            });
        
        return view('hod.marks.print-class', compact(
            'class',
            'units',
            'students',
            'studentMarks',
            'activeTerm',
            'institutionSettings',
            'hod'
        ));
    }
}
