<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification as NotificationFacade;

class NotificationController extends Controller
{
    /**
     * Display the notification creation form.
     */
    public function index()
    {
        $hod = Auth::user();

        if (!$hod->department_id) {
            return redirect()->route('hod.dashboard')
                ->with('warning', 'You are not assigned to a department.');
        }

        $activeTerm = \App\Services\ActiveTermService::getActiveTerm();

        // Get department classes for active term
        $classes = SchoolClass::where('department_id', $hod->department_id)
            ->when($activeTerm, function($query) use ($activeTerm) {
                $query->where('term_id', $activeTerm->id);
            })
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();

        // Get trainers in department
        $trainers = User::where('role', 'trainer')
            ->where('department_id', $hod->department_id)
            ->where('status', 'active')
            ->select('id', 'name', 'email')
            ->orderBy('name')
            ->get();

        // Get students in department
        $students = User::where('role', 'student')
            ->where('department_id', $hod->department_id)
            ->where('status', 'active')
            ->select('id', 'name', 'email')
            ->orderBy('name')
            ->get();

        return view('hod.notifications.index', compact('classes', 'trainers', 'students', 'activeTerm'));
    }

    /**
     * Send individual notification.
     */
    public function sendIndividual(Request $request)
    {
        $hod = Auth::user();

        $request->validate([
            'recipient_id' => ['required', 'exists:users,id'],
            'title' => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
        ]);

        $recipient = User::findOrFail($request->recipient_id);

        // Verify recipient is in HOD's department
        if ($recipient->department_id !== $hod->department_id) {
            return redirect()->back()
                ->with('error', 'You can only send notifications to users in your department.');
        }

        // Verify recipient is trainer or student
        if (!in_array($recipient->role, ['trainer', 'student'])) {
            return redirect()->back()
                ->with('error', 'You can only send notifications to trainers and students.');
        }

        // Create notification
        $recipient->notify(new \App\Notifications\GeneralNotification(
            $request->title,
            $request->message,
            $hod->id
        ));

        return redirect()->route('hod.notifications.index')
            ->with('success', 'Notification sent successfully.');
    }

    /**
     * Send bulk notification to department.
     */
    public function sendToDepartment(Request $request)
    {
        $hod = Auth::user();

        if (!$hod->department_id) {
            return redirect()->back()
                ->with('error', 'You are not assigned to a department.');
        }

        $request->validate([
            'title' => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
            'recipient_roles' => ['required', 'array', 'min:1'],
            'recipient_roles.*' => ['in:trainer,student'],
        ]);

        // Get users in department with specified roles
        $recipients = User::where('department_id', $hod->department_id)
            ->whereIn('role', $request->recipient_roles)
            ->where('status', 'active')
            ->get();

        if ($recipients->isEmpty()) {
            return redirect()->back()
                ->with('error', 'No recipients found matching the criteria.');
        }

        // Send notification to all recipients
        foreach ($recipients as $recipient) {
            $recipient->notify(new \App\Notifications\GeneralNotification(
                $request->title,
                $request->message,
                $hod->id
            ));
        }

        return redirect()->route('hod.notifications.index')
            ->with('success', "Notification sent to {$recipients->count()} recipient(s).");
    }

    /**
     * Send bulk notification to class.
     */
    public function sendToClass(Request $request)
    {
        $hod = Auth::user();

        $request->validate([
            'class_id' => ['required', 'exists:classes,id'],
            'title' => ['required', 'string', 'max:255'],
            'message' => ['required', 'string'],
            'recipient_roles' => ['required', 'array', 'min:1'],
            'recipient_roles.*' => ['in:trainer,student'],
        ]);

        $class = SchoolClass::findOrFail($request->class_id);

        // Verify class belongs to HOD's department
        if ($class->department_id !== $hod->department_id) {
            return redirect()->back()
                ->with('error', 'You can only send notifications to classes in your department.');
        }

        $recipients = collect();

        // Get students enrolled in the class
        if (in_array('student', $request->recipient_roles)) {
            $students = User::where('role', 'student')
                ->where('status', 'active')
                ->whereHas('enrollments', function($query) use ($class) {
                    $query->where('class_id', $class->id);
                })
                ->get();
            $recipients = $recipients->merge($students);
        }

        // Get trainers assigned to units in this class
        if (in_array('trainer', $request->recipient_roles)) {
            $trainerIds = DB::table('trainer_unit_class')
                ->where('class_id', $class->id)
                ->distinct()
                ->pluck('trainer_id');
            
            $trainers = User::whereIn('id', $trainerIds)
                ->where('status', 'active')
                ->get();
            $recipients = $recipients->merge($trainers);
        }

        $recipients = $recipients->unique('id');

        if ($recipients->isEmpty()) {
            return redirect()->back()
                ->with('error', 'No recipients found in this class.');
        }

        // Send notification to all recipients
        foreach ($recipients as $recipient) {
            $recipient->notify(new \App\Notifications\GeneralNotification(
                $request->title,
                $request->message,
                $hod->id
            ));
        }

        return redirect()->route('hod.notifications.index')
            ->with('success', "Notification sent to {$recipients->count()} recipient(s) in class {$class->name}.");
    }
}
