<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\SchoolClass;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ReportController extends Controller
{
    public function index()
    {
        $activeTerm = ActiveTermService::getActiveTerm();
        return view('hod.reports.index', compact('activeTerm'));
    }

    public function department(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || !$hod->department_id) {
            return redirect()->route('hod.reports.index')
                ->with('warning', 'No active term or department assigned.');
        }

        // Get department classes
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        // Get submissions for department
        $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('class_id', $departmentClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student', 'unit', 'schoolClass', 'validation'])
            ->get();

        return view('hod.reports.department', compact('submissions', 'activeTerm'));
    }

    public function class(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$activeTerm || !$hod->department_id) {
            return redirect()->route('hod.reports.index')
                ->with('warning', 'No active term or department assigned.');
        }

        $classId = $request->class;
        if (!$classId) {
            return redirect()->route('hod.reports.index')
                ->with('error', 'Please select a class.');
        }

        // Verify class belongs to HOD's department
        $class = SchoolClass::findOrFail($classId);
        if ($class->department_id !== $hod->department_id) {
            abort(403, 'You do not have permission to view this class report.');
        }

        // Get submissions for class
        $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('class_id', $classId)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student', 'unit', 'validation'])
            ->get();

        return view('hod.reports.class', compact('submissions', 'activeTerm', 'class'));
    }
}
