<?php

namespace App\Http\Controllers\HOD;

use App\Http\Controllers\Controller;
use App\Models\SchoolClass;
use App\Models\Unit;
use App\Models\User;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class UnitAllocationController extends Controller
{
    /**
     * Display a listing of unit allocations for department.
     */
    public function index(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$hod->department_id) {
            return view('hod.unit-allocations.index', [
                'allocations' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('warning', 'You are not assigned to a department.');
        }

        if (!$activeTerm) {
            return view('hod.unit-allocations.index', [
                'allocations' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found. Please set an active term to view unit allocations.');
        }

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        $query = DB::table('trainer_unit_class')
            ->join('users', 'trainer_unit_class.trainer_id', '=', 'users.id')
            ->join('units', 'trainer_unit_class.unit_id', '=', 'units.id')
            ->join('classes', 'trainer_unit_class.class_id', '=', 'classes.id')
            ->where('units.department_id', $hod->department_id)
            ->whereIn('classes.id', $departmentClassIds)
            ->select(
                'trainer_unit_class.id',
                'trainer_unit_class.trainer_id',
                'trainer_unit_class.unit_id',
                'trainer_unit_class.class_id',
                'users.name as trainer_name',
                'users.email as trainer_email',
                'units.name as unit_name',
                'units.code as unit_code',
                'classes.name as class_name',
                'classes.code as class_code',
                'trainer_unit_class.created_at'
            );

        // Filter by class
        if ($request->has('class') && $request->class) {
            $query->where('classes.id', $request->class);
        }

        // Filter by trainer
        if ($request->has('trainer') && $request->trainer) {
            $query->where('users.id', $request->trainer);
        }

        // Filter by unit
        if ($request->has('unit') && $request->unit) {
            $query->where('units.id', $request->unit);
        }

        $allocations = $query->orderBy('trainer_unit_class.created_at', 'desc')->paginate(20);

        // Get filter options (department-scoped)
        $classes = SchoolClass::whereIn('id', $departmentClassIds)
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();
        $trainers = User::where('role', 'trainer')
            ->where('department_id', $hod->department_id)
            ->where('status', 'active')
            ->select('id', 'name', 'email')
            ->orderBy('name')
            ->get();
        $units = Unit::where('department_id', $hod->department_id)
            ->select('id', 'name', 'code')
            ->orderBy('name')
            ->get();

        return view('hod.unit-allocations.index', compact(
            'allocations',
            'classes',
            'trainers',
            'units',
            'activeTerm'
        ));
    }

    /**
     * Show the form for creating a new allocation.
     */
    public function create(Request $request)
    {
        $hod = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();

        if (!$hod->department_id) {
            return redirect()->route('hod.unit-allocations.index')
                ->with('warning', 'You are not assigned to a department.');
        }

        // Get department classes for active term
        $departmentClassIds = SchoolClass::where('term_id', $activeTerm->id)
            ->where('department_id', $hod->department_id)
            ->pluck('id')
            ->toArray();

        $classes = SchoolClass::whereIn('id', $departmentClassIds)
            ->with(['term:id,name', 'level:id,name'])
            ->orderBy('name')
            ->get();

        $trainers = User::where('department_id', $hod->department_id)
            ->where(function($query) {
                $query->where('role', 'trainer')
                      ->orWhere('role', 'hod'); // Include HOD as they can also be allocated units
            })
            ->where('status', 'active')
            ->orderBy('name')
            ->get();

        $units = Unit::where('department_id', $hod->department_id)
            ->orderBy('name')
            ->get();

        return view('hod.unit-allocations.create', compact('classes', 'trainers', 'units', 'activeTerm'));
    }

    /**
     * Store a newly created allocation.
     */
    public function store(Request $request)
    {
        $hod = Auth::user();

        if (!$hod->department_id) {
            return redirect()->back()
                ->with('error', 'You are not assigned to a department.');
        }

        $validator = Validator::make($request->all(), [
            'trainer_id' => 'required|exists:users,id',
            'unit_id' => 'required|array|min:1',
            'unit_id.*' => 'exists:units,id',
            'class_id' => 'required|exists:classes,id',
        ]);

        // Verify trainer/HOD belongs to HOD's department
        $trainer = User::findOrFail($request->trainer_id);
        if ($trainer->department_id !== $hod->department_id || !in_array($trainer->role, ['trainer', 'hod'])) {
            $validator->errors()->add('trainer_id', 'Invalid trainer/HOD selected.');
        }

        // Verify class belongs to HOD's department
        $class = SchoolClass::findOrFail($request->class_id);
        if ($class->department_id !== $hod->department_id) {
            $validator->errors()->add('class_id', 'Invalid class selected.');
        }
        
        // Get term_id from the class (required for trainer_unit_class)
        $termId = $class->term_id;
        if (!$termId) {
            return redirect()->back()
                ->with('error', 'The selected class does not have a term assigned.')
                ->withInput();
        }

        // Verify units belong to HOD's department
        $validUnits = Unit::whereIn('id', $request->unit_id)
            ->where('department_id', $hod->department_id)
            ->get();
        
        if ($validUnits->count() !== count($request->unit_id)) {
            $validator->errors()->add('unit_id', 'One or more units are invalid.');
        }

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $successCount = 0;
        $errorMessages = [];

        DB::beginTransaction();
        try {
            foreach ($request->unit_id as $unitId) {
                // Check if allocation already exists
                $exists = DB::table('trainer_unit_class')
                    ->where('trainer_id', $request->trainer_id)
                    ->where('unit_id', $unitId)
                    ->where('class_id', $request->class_id)
                    ->exists();

                if ($exists) {
                    $unit = Unit::find($unitId);
                    $errorMessages[] = "Trainer {$trainer->name} is already assigned to unit {$unit->name} in class {$class->name}.";
                    continue;
                }

                DB::table('trainer_unit_class')->insert([
                    'trainer_id' => $request->trainer_id,
                    'unit_id' => $unitId,
                    'class_id' => $request->class_id,
                    'term_id' => $termId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                $successCount++;
            }

            DB::commit();

            if ($successCount > 0) {
                $message = "Successfully assigned {$successCount} unit(s) to {$trainer->name}.";
                if (!empty($errorMessages)) {
                    $message .= " " . implode(' ', $errorMessages);
                }
                return redirect()->route('hod.unit-allocations.index')
                    ->with('success', $message);
            } else {
                return redirect()->back()
                    ->with('error', implode(' ', $errorMessages))
                    ->withInput();
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to assign units: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Allocate units to a class.
     */
    public function allocateToClass(Request $request)
    {
        $hod = Auth::user();

        if (!$hod->department_id) {
            return redirect()->back()
                ->with('error', 'You are not assigned to a department.');
        }

        $validator = Validator::make($request->all(), [
            'class_id' => 'required|exists:classes,id',
            'unit_ids' => 'required|array|min:1',
            'unit_ids.*' => 'exists:units,id',
        ]);

        // Verify class belongs to HOD's department
        $class = SchoolClass::findOrFail($request->class_id);
        if ($class->department_id !== $hod->department_id) {
            $validator->errors()->add('class_id', 'Invalid class selected.');
        }

        // Verify units belong to HOD's department
        $validUnits = Unit::whereIn('id', $request->unit_ids)
            ->where('department_id', $hod->department_id)
            ->get();
        
        if ($validUnits->count() !== count($request->unit_ids)) {
            $validator->errors()->add('unit_ids', 'One or more units are invalid.');
        }

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $successCount = 0;
        $errorMessages = [];

        DB::beginTransaction();
        try {
            foreach ($request->unit_ids as $unitId) {
                // Check if unit is already assigned to class
                if ($class->units()->where('unit_id', $unitId)->exists()) {
                    $unit = Unit::find($unitId);
                    $errorMessages[] = "Unit {$unit->name} is already assigned to class {$class->name}.";
                    continue;
                }

                $class->units()->attach($unitId);
                $successCount++;
            }

            DB::commit();

            if ($successCount > 0) {
                $message = "Successfully assigned {$successCount} unit(s) to class {$class->name}.";
                if (!empty($errorMessages)) {
                    $message .= " " . implode(' ', $errorMessages);
                }
                return redirect()->route('hod.classes.show', $class)
                    ->with('success', $message);
            } else {
                return redirect()->back()
                    ->with('error', implode(' ', $errorMessages))
                    ->withInput();
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'Failed to assign units: ' . $e->getMessage())
                ->withInput();
        }
    }
}
