<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\SchoolClass;
use App\Models\PoeSubmission;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ClassController extends Controller
{
    /**
     * Display a listing of assigned classes.
     */
    public function index()
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('trainer.classes.index', [
                'classes' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found.');
        }

        // Get classes assigned to this trainer via trainer_unit_class
        $assignedClassIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->distinct()
            ->pluck('class_id');

        // Get classes with their details
        $classes = SchoolClass::whereIn('id', $assignedClassIds)
            ->where('term_id', $activeTerm->id)
            ->with(['department:id,name,code', 'level:id,name', 'term:id,name'])
            ->withCount('enrollments')
            ->get()
            ->map(function($class) use ($trainer, $activeTerm) {
                // Get units assigned to this trainer for this class
                $unitIds = DB::table('trainer_unit_class')
                    ->where('trainer_id', $trainer->id)
                    ->where('class_id', $class->id)
                    ->pluck('unit_id');
                
                $class->assigned_units = \App\Models\Unit::whereIn('id', $unitIds)
                    ->select('id', 'name', 'code')
                    ->get();
                
                // Count submissions for assigned units in this class
                $class->submissions_count = PoeSubmission::withoutGlobalScope('activeTerm')
                    ->where('class_id', $class->id)
                    ->whereIn('unit_id', $unitIds)
                    ->whereHas('schoolClass', function($q) use ($activeTerm) {
                        $q->where('term_id', $activeTerm->id);
                    })
                    ->count();
                
                return $class;
            })
            ->sortBy('name')
            ->values();

        return view('trainer.classes.index', compact('classes', 'activeTerm'));
    }

    /**
     * Display the specified class.
     */
    public function show(SchoolClass $class)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        // Verify trainer is assigned to this class
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $class->id)
            ->exists();
        
        if (!$isAssigned) {
            abort(403, 'You are not assigned to this class.');
        }

        // Get assigned units for this class
        $unitIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('class_id', $class->id)
            ->pluck('unit_id');
        
        $assignedUnits = \App\Models\Unit::whereIn('id', $unitIds)
            ->with('department:id,name')
            ->get();
        
        // Get students enrolled in this class
        $students = $class->students()
            ->with('enrollments', function($query) use ($class) {
                $query->where('class_id', $class->id);
            })
            ->get();
        
        // Get submissions for assigned units
        $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('class_id', $class->id)
            ->whereIn('unit_id', $unitIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student:id,name,email', 'unit:id,name,code'])
            ->latest()
            ->paginate(20);

        $class->load(['department:id,name,code', 'level:id,name', 'term:id,name']);

        return view('trainer.classes.show', compact('class', 'assignedUnits', 'students', 'submissions', 'activeTerm'));
    }
}
