<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\Assignment;
use App\Models\SchoolClass;
use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PracticalMarksController extends Controller
{
    /**
     * Display compiled practical marks for an assignment
     */
    public function index(Request $request, Assignment $assignment)
    {
        $trainer = Auth::user();
        
        // Verify trainer owns this assignment
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403, 'You do not have permission to view this assignment.');
        }

        if ($assignment->type !== 'practical') {
            abort(404, 'This is not a practical assignment.');
        }

        // Get students enrolled in the specific class this assignment belongs to
        $unit = $assignment->unit;
        
        // CRITICAL: Only get students from the assignment's specific class, not all classes with the unit
        if (!$assignment->class_id) {
            return back()->with('error', 'This assignment is not linked to a specific class.');
        }

        // Load the class once (all students will be from this class)
        $class = SchoolClass::find($assignment->class_id);
        if (!$class) {
            return back()->with('error', 'The class for this assignment was not found.');
        }

        // Get students enrolled in this specific class
        $students = DB::table('enrollments')
            ->where('enrollments.class_id', $assignment->class_id) // Only students from assignment's class
            ->where('enrollments.status', 'active')
            ->join('users', 'enrollments.student_id', '=', 'users.id')
            ->where('users.role', 'student')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number', 'enrollments.class_id')
            ->orderBy('users.name')
            ->get();

        // Get submissions and marks for each student
        $compiledMarks = [];
        foreach ($students as $student) {
            $submission = \App\Models\AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            $marks = 0;
            $markingStatus = 'not_marked';
            
            if ($submission) {
                $markingStatus = $submission->marking_status ?? 'not_marked';
                if ($submission->marks) {
                    $marks = $submission->marks;
                } elseif ($submission->assessorToolMarks()->count() > 0) {
                    // Calculate from assessor tool marks
                    $marks = $submission->assessorToolMarks()->sum('marks_obtained');
                }
            }

            $compiledMarks[] = [
                'student' => $student,
                'submission' => $submission,
                'marks' => $marks,
                'marking_status' => $markingStatus,
                'class' => $class, // Reuse the same class object for all students
            ];
        }

        // Filter by status
        $statusFilter = $request->get('status', 'all');
        if ($statusFilter !== 'all') {
            $compiledMarks = array_filter($compiledMarks, function($item) use ($statusFilter) {
                return $item['marking_status'] === $statusFilter;
            });
        }

        // Sort
        $sortBy = $request->get('sort', 'name');
        if ($sortBy === 'marks') {
            usort($compiledMarks, function($a, $b) {
                return $b['marks'] <=> $a['marks'];
            });
        } else {
            usort($compiledMarks, function($a, $b) {
                return strcmp($a['student']->name, $b['student']->name);
            });
        }

        // Statistics
        $marked = count(array_filter($compiledMarks, fn($item) => $item['marking_status'] === 'marked'));
        $absent = count(array_filter($compiledMarks, fn($item) => $item['marking_status'] === 'absent'));
        $notMarked = count(array_filter($compiledMarks, fn($item) => $item['marking_status'] === 'not_marked'));
        $assessed = $marked + $absent; // Assessed = marked + absent
        
        $stats = [
            'total_students' => count($compiledMarks),
            'marked' => $marked,
            'not_marked' => $notMarked,
            'absent' => $absent,
            'assessed' => $assessed, // Total assessed (marked + absent)
        ];

        return view('trainer.practical-marks.index', compact(
            'assignment',
            'unit',
            'compiledMarks',
            'stats',
            'statusFilter',
            'sortBy'
        ));
    }

    /**
     * Mark student as absent
     */
    public function markAbsent(Request $request, Assignment $assignment)
    {
        $trainer = Auth::user();
        
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403);
        }

        $request->validate([
            'student_id' => 'required|exists:users,id',
        ]);

        try {
            $submission = \App\Models\AssignmentSubmission::firstOrCreate([
                'assignment_id' => $assignment->id,
                'student_id' => $request->student_id,
            ], [
                'status' => 'submitted',
                'marking_status' => 'absent',
                'marks' => 0,
            ]);

            $submission->update([
                'marking_status' => 'absent',
                'marks' => 0,
            ]);

            // Return JSON response for AJAX requests
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Student marked as assessed but absent (AB).',
                ]);
            }

            return redirect()->back()->with('success', 'Student marked as assessed but absent (AB).');
        } catch (\Exception $e) {
            \Log::error('Error marking student as absent: ' . $e->getMessage());
            
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error marking student as absent: ' . $e->getMessage(),
                ], 500);
            }

            return redirect()->back()->with('error', 'Error marking student as absent. Please try again.');
        }
    }

    /**
     * Remove absent status
     */
    public function removeAbsent(Request $request, Assignment $assignment)
    {
        $trainer = Auth::user();
        
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403);
        }

        $request->validate([
            'student_id' => 'required|exists:users,id',
        ]);

        try {
            $submission = \App\Models\AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $request->student_id)
                ->first();

            if ($submission) {
                $submission->update([
                    'marking_status' => 'not_marked',
                    'marks' => null,
                ]);
            }

            // Return JSON response for AJAX requests
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Absent status removed.',
                ]);
            }

            return redirect()->back()->with('success', 'Absent status removed.');
        } catch (\Exception $e) {
            \Log::error('Error removing absent status: ' . $e->getMessage());
            
            if ($request->expectsJson() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error removing absent status: ' . $e->getMessage(),
                ], 500);
            }

            return redirect()->back()->with('error', 'Error removing absent status. Please try again.');
        }
    }

    /**
     * Export practical marks
     */
    public function export(Assignment $assignment, $format)
    {
        $trainer = Auth::user();
        
        if ($assignment->trainer_id !== $trainer->id) {
            abort(403);
        }

        if ($assignment->type !== 'practical') {
            abort(404);
        }

        // Get compiled marks (same logic as index) - only students from assignment's specific class
        $unit = $assignment->unit;
        
        // CRITICAL: Only get students from the assignment's specific class
        if (!$assignment->class_id) {
            abort(400, 'This assignment is not linked to a specific class.');
        }

        $students = DB::table('enrollments')
            ->where('enrollments.class_id', $assignment->class_id) // Only students from assignment's class
            ->where('enrollments.status', 'active')
            ->join('users', 'enrollments.student_id', '=', 'users.id')
            ->where('users.role', 'student')
            ->select('users.id', 'users.name', 'users.email', 'users.admission_number', 'enrollments.class_id')
            ->orderBy('users.name')
            ->get();

        $compiledMarks = [];
        foreach ($students as $student) {
            $submission = \App\Models\AssignmentSubmission::where('assignment_id', $assignment->id)
                ->where('student_id', $student->id)
                ->first();

            $marks = 0;
            $markingStatus = 'not_marked';
            
            if ($submission) {
                $markingStatus = $submission->marking_status ?? 'not_marked';
                if ($submission->marks) {
                    $marks = $submission->marks;
                } elseif ($submission->assessorToolMarks()->count() > 0) {
                    $marks = $submission->assessorToolMarks()->sum('marks_obtained');
                }
            }

            $compiledMarks[] = [
                'student' => $student,
                'marks' => $marks,
                'marking_status' => $markingStatus,
                'class' => SchoolClass::find($student->class_id),
            ];
        }

        // Sort by name
        usort($compiledMarks, function($a, $b) {
            return strcmp($a['student']->name, $b['student']->name);
        });

        if ($format === 'pdf') {
            return $this->exportPDF($assignment, $unit, $compiledMarks);
        } elseif ($format === 'excel') {
            return $this->exportExcel($assignment, $unit, $compiledMarks);
        } elseif ($format === 'word') {
            return $this->exportWord($assignment, $unit, $compiledMarks);
        }

        abort(404, 'Invalid export format');
    }

    protected function exportPDF($assignment, $unit, $compiledMarks)
    {
        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('trainer.practical-marks.export-pdf', [
            'assignment' => $assignment,
            'unit' => $unit,
            'compiledMarks' => $compiledMarks,
            'trainer' => Auth::user(),
        ]);
        
        $filename = 'Practical_Marks_' . $assignment->title . '_' . now()->format('Y-m-d') . '.pdf';
        return $pdf->download($filename);
    }

    protected function exportExcel($assignment, $unit, $compiledMarks)
    {
        // For Excel, we'll use a simple CSV for now
        // You can install maatwebsite/excel for better Excel support
        $filename = 'Practical_Marks_' . $assignment->title . '_' . now()->format('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($assignment, $unit, $compiledMarks) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, ['#', 'Student Name', 'Admission Number', 'Class', 'Status', 'Marks', 'Total Marks']);
            
            // Data rows
            foreach ($compiledMarks as $index => $item) {
                fputcsv($file, [
                    $index + 1,
                    $item['student']->name,
                    $item['student']->admission_number ?? 'N/A',
                    $item['class']->name ?? 'N/A',
                    ucfirst(str_replace('_', ' ', $item['marking_status'])),
                    number_format($item['marks'], 2),
                    number_format($assignment->total_marks, 2),
                ]);
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    protected function exportWord($assignment, $unit, $compiledMarks)
    {
        // Word export can be implemented using PhpWord
        // For now, redirect to PDF
        return $this->exportPDF($assignment, $unit, $compiledMarks);
    }
}
