<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;

class ProfileController extends Controller
{
    /**
     * Display the trainer's profile.
     */
    public function show()
    {
        $user = Auth::user();
        $user->load('department:id,name,code');
        
        // Get assigned units
        $assignedUnits = $user->units()->with('department:id,name')->get();
        
        // Get assigned classes (from trainer_unit_class)
        $assignedClasses = \Illuminate\Support\Facades\DB::table('trainer_unit_class')
            ->join('classes', 'trainer_unit_class.class_id', '=', 'classes.id')
            ->join('units', 'trainer_unit_class.unit_id', '=', 'units.id')
            ->where('trainer_unit_class.trainer_id', $user->id)
            ->select(
                'classes.id as class_id',
                'classes.name as class_name',
                'classes.code as class_code',
                'units.id as unit_id',
                'units.name as unit_name',
                'units.code as unit_code'
            )
            ->get()
            ->groupBy('class_id');
        
        return view('trainer.profile.show', compact('user', 'assignedUnits', 'assignedClasses'));
    }

    /**
     * Show the form for editing the profile.
     */
    public function edit()
    {
        $user = Auth::user();
        $user->load('department:id,name,code');
        
        return view('trainer.profile.edit', compact('user'));
    }

    /**
     * Update the trainer's profile.
     */
    public function update(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
            'phone' => ['nullable', 'string', 'max:20'],
            'current_password' => ['nullable', 'required_with:password', 'current-password'],
            'password' => ['nullable', 'confirmed', Rules\Password::defaults()],
        ]);

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone = $request->phone;

        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }

        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
        }

        $user->save();

        return redirect()->route('trainer.profile.show')
            ->with('success', 'Profile updated successfully.');
    }
}
