<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\PoeSubmission;
use App\Models\AssignmentSubmission;
use App\Models\Assignment;
use App\Services\ActiveTermService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Pagination\LengthAwarePaginator;

class SubmissionController extends Controller
{
    /**
     * Display all submissions for assigned units (combines PoE and Assignment submissions).
     */
    public function index(Request $request)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('trainer.submissions.index', [
                'submissions' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found.');
        }

        // Get unit IDs assigned to this trainer
        $unitIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->distinct()
            ->pluck('unit_id');
        
        // Get class IDs assigned to this trainer
        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->distinct()
            ->pluck('class_id');

        // Get PoE submissions
        $poeQuery = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $classIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student:id,name,email', 'unit:id,name,code', 'schoolClass:id,name,code']);

        // Get Assignment submissions
        $assignmentQuery = AssignmentSubmission::whereHas('assignment', function($q) use ($trainer, $unitIds) {
                $q->where('trainer_id', $trainer->id)
                  ->whereIn('unit_id', $unitIds);
            })
            ->with([
                'student:id,name,email',
                'assignment:id,title,type,unit_id,total_marks',
                'assignment.unit:id,name,code'
            ])
            ->whereNotNull('assignment_id')
            ->whereHas('student.enrollments', function($q) use ($classIds, $activeTerm) {
                $q->whereIn('class_id', $classIds)
                  ->where('status', 'active')
                  ->whereHas('schoolClass', function($qc) use ($activeTerm) {
                      $qc->where('term_id', $activeTerm->id);
                  });
            });

        // Apply filters to PoE submissions
        if ($request->has('status') && $request->status) {
            $poeQuery->where('status', $request->status);
        }

        if ($request->has('unit') && $request->unit) {
            $poeQuery->where('unit_id', $request->unit);
        }

        if ($request->has('class') && $request->class) {
            $poeQuery->where('class_id', $request->class);
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $poeQuery->whereHas('student', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Apply filters to Assignment submissions
        if ($request->has('status') && $request->status) {
            // Map PoE status to assignment status
            $statusMap = [
                'draft' => 'draft',
                'submitted' => 'submitted',
                'trainer_approved' => 'graded',
            ];
            if (isset($statusMap[$request->status])) {
                $assignmentQuery->where('status', $statusMap[$request->status]);
            }
        }

        if ($request->has('unit') && $request->unit) {
            $assignmentQuery->whereHas('assignment', function($q) use ($request) {
                $q->where('unit_id', $request->unit);
            });
        }

        if ($request->has('class') && $request->class) {
            $assignmentQuery->whereHas('student.enrollments', function($q) use ($request) {
                $q->where('class_id', $request->class)
                  ->where('status', 'active');
            });
        }

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $assignmentQuery->whereHas('student', function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Get both types of submissions
        $poeSubmissions = $poeQuery->get()->map(function($submission) {
            $submission->submission_type = 'poe';
            $submission->display_submitted_at = $submission->submitted_at;
            return $submission;
        });

        $assignmentSubmissions = $assignmentQuery->get()->map(function($submission) use ($activeTerm) {
            // Get class_id from student's enrollment
            $enrollment = $submission->student->enrollments()
                ->where('status', 'active')
                ->whereHas('schoolClass', function($q) use ($activeTerm) {
                    $q->where('term_id', $activeTerm->id);
                })
                ->first();
            
            $submission->submission_type = 'assignment';
            $submission->unit_id = $submission->assignment->unit_id;
            $submission->class_id = $enrollment ? $enrollment->class_id : null;
            $submission->unit = $submission->assignment->unit;
            $submission->schoolClass = $enrollment ? $enrollment->schoolClass : null;
            $submission->display_submitted_at = $submission->submitted_at;
            return $submission;
        });

        // Combine and sort by submitted_at
        $allSubmissions = $poeSubmissions->concat($assignmentSubmissions)
            ->sortByDesc('display_submitted_at')
            ->values();

        // Paginate manually
        $page = $request->get('page', 1);
        $perPage = 20;
        $currentItems = $allSubmissions->slice(($page - 1) * $perPage, $perPage)->all();
        $submissions = new LengthAwarePaginator(
            $currentItems,
            $allSubmissions->count(),
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        // Get filter options
        $units = \App\Models\Unit::whereIn('id', $unitIds)->orderBy('name')->get();
        $classes = \App\Models\SchoolClass::whereIn('id', $classIds)
            ->where('term_id', $activeTerm->id)
            ->orderBy('name')
            ->get();

        return view('trainer.submissions.index', compact('submissions', 'units', 'classes', 'activeTerm'));
    }

    /**
     * Display the specified submission.
     */
    public function show(PoeSubmission $submission)
    {
        $trainer = Auth::user();
        
        // Verify trainer is assigned
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $submission->unit_id)
            ->where('class_id', $submission->class_id)
            ->exists();
        
        if (!$isAssigned) {
            abort(403, 'You are not assigned to view this submission.');
        }

        $submission->load([
            'student:id,name,email,admission_number',
            'unit:id,name,code,description',
            'schoolClass:id,name,code',
            'evidence',
        ]);

        return view('trainer.submissions.show', compact('submission'));
    }
}
