<?php

namespace App\Http\Controllers\Trainer;

use App\Http\Controllers\Controller;
use App\Models\Unit;
use App\Models\Assignment;
use App\Models\AssignmentSubmission;
use App\Models\PoeSubmission;
use App\Services\ActiveTermService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UnitController extends Controller
{
    /**
     * Display a listing of assigned units.
     */
    public function index()
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        if (!$activeTerm) {
            return view('trainer.units.index', [
                'units' => collect(),
                'activeTerm' => null,
            ])->with('warning', 'No active term found.');
        }

        // First, get active term classes
        $activeTermClassIds = \App\Models\SchoolClass::where('term_id', $activeTerm->id)
            ->pluck('id')
            ->toArray();

        // Get trainer-unit-class allocations for the active term
        $allocations = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->select('unit_id', 'class_id')
            ->get();

        // If no units assigned in active term, return empty
        if ($allocations->isEmpty()) {
            return view('trainer.units.index', [
                'units' => collect(),
                'activeTerm' => $activeTerm,
            ])->with('info', 'You don\'t have any units assigned for the current term.');
        }

        $unitIds = $allocations->pluck('unit_id')->unique()->toArray();

        // Build unit -> [class_ids] map
        $unitClassMap = $allocations
            ->groupBy('unit_id')
            ->map(function ($items) {
                return $items->pluck('class_id')->toArray();
            });
        
        $allClassIds = $unitClassMap->flatten()->unique()->toArray();
        
        // Batch load classes - already filtered by active term
        $classes = \App\Models\SchoolClass::whereIn('id', $allClassIds)
            ->where('term_id', $activeTerm->id)
            ->select('id', 'name', 'code')
            ->get()
            ->keyBy('id');
        
        // Batch load students counts per class
        $studentsCounts = \Illuminate\Support\Facades\DB::table('enrollments')
            ->join('users', 'enrollments.student_id', '=', 'users.id')
            ->where('users.role', 'student')
            ->whereIn('enrollments.class_id', $allClassIds)
            ->selectRaw('enrollments.class_id, count(distinct enrollments.student_id) as count')
            ->groupBy('enrollments.class_id')
            ->pluck('count', 'class_id');
        
        // Calculate students per unit
        $unitStudentsCount = [];
        foreach ($unitClassMap as $unitId => $classIds) {
            $unitStudentsCount[$unitId] = collect($classIds)->sum(function($classId) use ($studentsCounts) {
                return $studentsCounts[$classId] ?? 0;
            });
        }
        
        // Batch load assignments counts - filtered by active term AND trainer's assigned classes
        $assignmentsCounts = Assignment::whereIn('unit_id', $unitIds)
            ->where('trainer_id', $trainer->id)
            ->whereIn('class_id', $allClassIds) // CRITICAL: Only assignments for trainer's assigned classes
            ->where('created_at', '>=', $activeTerm->start_date)
            ->where('created_at', '<=', $activeTerm->end_date ?? now())
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');
        
        // Batch load PoE submissions counts
        $poeSubmissionsCounts = PoeSubmission::withoutGlobalScope('activeTerm')
            ->whereIn('unit_id', $unitIds)
            ->whereIn('class_id', $allClassIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->selectRaw('unit_id, count(*) as count')
            ->groupBy('unit_id')
            ->pluck('count', 'unit_id');
        
        // Batch load Assignment submissions counts
        $assignmentSubmissionsCounts = AssignmentSubmission::whereHas('assignment', function($q) use ($trainer, $unitIds) {
                $q->where('trainer_id', $trainer->id)
                  ->whereIn('unit_id', $unitIds);
            })
            ->whereHas('student.enrollments', function($q) use ($allClassIds, $activeTerm) {
                $q->whereIn('class_id', $allClassIds)
                  ->where('status', 'active')
                  ->whereHas('schoolClass', function($qc) use ($activeTerm) {
                      $qc->where('term_id', $activeTerm->id);
                  });
            })
            ->whereHas('assignment.unit', function($q) use ($unitIds) {
                $q->whereIn('id', $unitIds);
            })
            ->selectRaw('assignments.unit_id, count(*) as count')
            ->join('assignments', 'assignment_submissions.assignment_id', '=', 'assignments.id')
            ->groupBy('assignments.unit_id')
            ->pluck('count', 'unit_id');
        
        // Combine both counts
        $submissionsCounts = [];
        foreach ($unitIds as $unitId) {
            $submissionsCounts[$unitId] = ($poeSubmissionsCounts[$unitId] ?? 0) + ($assignmentSubmissionsCounts[$unitId] ?? 0);
        }

        // Load base unit models
        $baseUnits = Unit::whereIn('id', $unitIds)
            ->with(['department:id,name,code'])
            ->get()
            ->keyBy('id');

        // Build view models: one card PER (unit, class) allocation
        $units = $allocations->map(function ($row) use (
            $baseUnits,
            $classes,
            $unitStudentsCount,
            $assignmentsCounts,
            $submissionsCounts
        ) {
            $unit = $baseUnits->get($row->unit_id);
            if (! $unit) {
                return null;
            }

            // Clone so each allocation can carry its own class info
            $clone = clone $unit;

            $class = $classes->get($row->class_id);
            $clone->assigned_classes = $class ? collect([$class]) : collect();
            $clone->allocation_class_id = $row->class_id;

            // Counts are still per unit (aggregated across classes) – cheap and good enough
            $clone->students_count = $unitStudentsCount[$unit->id] ?? 0;
            $clone->assignments_count = $assignmentsCounts[$unit->id] ?? 0;
            $clone->submissions_count = $submissionsCounts[$unit->id] ?? 0;

            return $clone;
        })
        ->filter()
        ->sortBy(function ($u) {
            $className = optional($u->assigned_classes->first())->name ?? '';
            return $u->name.' '.$className;
        })
        ->values();

        return view('trainer.units.index', compact('units', 'activeTerm'));
    }

    /**
     * Display the specified unit.
     */
    public function show(Unit $unit)
    {
        $trainer = Auth::user();
        $activeTerm = ActiveTermService::getActiveTerm();
        
        // Verify trainer is assigned to this unit
        $isAssigned = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unit->id)
            ->exists();
        
        if (!$isAssigned) {
            abort(403, 'You are not assigned to this unit.');
        }

        // Get classes where this unit is assigned to this trainer - ONLY for active term
        $activeTermClassIds = \App\Models\SchoolClass::where('term_id', $activeTerm->id)
            ->pluck('id')
            ->toArray();

        $classIds = DB::table('trainer_unit_class')
            ->where('trainer_id', $trainer->id)
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $activeTermClassIds)
            ->pluck('class_id')
            ->toArray();

        // If a specific class_id was requested, narrow down to that class only
        $requestedClassId = request()->query('class_id');
        if ($requestedClassId) {
            $requestedClassId = (int) $requestedClassId;
            if (in_array($requestedClassId, $classIds, true)) {
                $classIds = [$requestedClassId];
            } else {
                abort(403, 'You are not assigned to this unit in that class for the current term.');
            }
        }

        if (empty($classIds)) {
            abort(403, 'You are not assigned to this unit in the current term.');
        }
        
        $assignedClasses = \App\Models\SchoolClass::whereIn('id', $classIds)
            ->where('term_id', $activeTerm->id)
            ->with(['department:id,name', 'level:id,name'])
            ->withCount('enrollments')
            ->get();
        
        // Get students enrolled in these classes
        $students = \App\Models\User::where('role', 'student')
            ->whereHas('enrollments', function($query) use ($classIds) {
                $query->whereIn('class_id', $classIds);
            })
            ->with(['enrollments' => function($query) use ($classIds) {
                $query->whereIn('class_id', $classIds)->with('schoolClass:id,name');
            }])
            ->distinct()
            ->get();
        
        // Get assignments for this unit - filtered by active term AND specific classes
        // CRITICAL: Only show assignments created by THIS trainer for the specific classes they're assigned to
        // This ensures sofia (ITECH6M26) doesn't see assignments from martin (ICT4M26)
        // Each trainer only sees their own assignments for their assigned classes
        $assignments = Assignment::where('unit_id', $unit->id)
            ->where('trainer_id', $trainer->id) // STRICT: Only this trainer's assignments
            ->whereIn('class_id', $classIds) // CRITICAL: Only assignments for trainer's assigned classes
            ->where('created_at', '>=', $activeTerm->start_date)
            ->where('created_at', '<=', $activeTerm->end_date ?? now())
            ->latest()
            ->paginate(10);
        
        // Get submissions for this unit
        $submissions = PoeSubmission::withoutGlobalScope('activeTerm')
            ->where('unit_id', $unit->id)
            ->whereIn('class_id', $classIds)
            ->whereHas('schoolClass', function($q) use ($activeTerm) {
                $q->where('term_id', $activeTerm->id);
            })
            ->with(['student:id,name,email', 'schoolClass:id,name'])
            ->latest()
            ->paginate(20);

        $unit->load('department:id,name,code');

        return view('trainer.units.show', compact('unit', 'assignedClasses', 'students', 'assignments', 'submissions', 'activeTerm'));
    }
}
