<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;

class Term extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'start_date',
        'end_date',
        'status',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    // Relationships
    public function classes()
    {
        return $this->hasMany(SchoolClass::class);
    }

    public function termDepartmentUnits()
    {
        return $this->hasMany(TermDepartmentUnit::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeArchived($query)
    {
        return $query->where('status', 'archived');
    }

    public function scopeClosed($query)
    {
        return $query->whereIn('status', ['completed', 'archived']);
    }

    // Static helper to get active term
    public static function getActive()
    {
        return static::where('status', 'active')->first();
    }

    // Helper methods
    public function isActive()
    {
        return $this->status === 'active';
    }

    public function isDraft()
    {
        return $this->status === 'draft';
    }

    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    public function isArchived()
    {
        return $this->status === 'archived';
    }

    public function isClosed()
    {
        return in_array($this->status, ['completed', 'archived']);
    }

    public function isReadOnly()
    {
        return $this->isClosed();
    }

    // Activate this term (closes previous active term)
    public function activate()
    {
        // Close any currently active term
        static::where('status', 'active')
            ->where('id', '!=', $this->id)
            ->update(['status' => 'completed']);

        // Activate this term
        $this->update(['status' => 'active']);
        
        return $this;
    }

    // Close this term
    public function close()
    {
        if ($this->isActive()) {
            $this->update(['status' => 'completed']);
        }
        return $this;
    }

    // Get duration in days
    public function getDurationInDays()
    {
        return $this->start_date->diffInDays($this->end_date);
    }

    // Check if term is currently within date range
    public function isCurrentlyActive()
    {
        $now = Carbon::now();
        return $this->status === 'active' 
            && $now->greaterThanOrEqualTo($this->start_date) 
            && $now->lessThanOrEqualTo($this->end_date);
    }
}
