<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'first_name',
        'last_name',
        'admission_number',
        'email',
        'password',
        'role',
        'department_id',
        'phone',
        'avatar',
        'status',
        // Student profile fields
        'national_id_number',
        'date_of_birth',
        'guardian_name',
        'guardian_phone',
        'address',
        'id_document_path',
        'kcpe_certificate_path',
        'kcse_certificate_path',
        'birth_certificate_path',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    // Relationships
    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    public function enrollments()
    {
        return $this->hasMany(Enrollment::class, 'student_id');
    }

    public function classes()
    {
        return $this->belongsToMany(SchoolClass::class, 'enrollments', 'student_id', 'class_id')
                    ->withPivot('enrolled_at', 'status')
                    ->withTimestamps();
    }

    public function poeSubmissions()
    {
        return $this->hasMany(PoeSubmission::class, 'student_id');
    }

    public function validations()
    {
        return $this->hasMany(Validation::class, 'validator_id');
    }

    public function sentMessages()
    {
        return $this->hasMany(Message::class, 'sender_id');
    }

    public function receivedMessages()
    {
        return $this->hasMany(Message::class, 'receiver_id');
    }

    public function auditLogs()
    {
        return $this->hasMany(AuditLog::class);
    }

    public function units()
    {
        return $this->belongsToMany(Unit::class, 'trainer_unit', 'trainer_id', 'unit_id')
                    ->withTimestamps();
    }

    // Role check methods
    public function isAdmin()
    {
        return $this->role === 'admin';
    }

    public function isHOD()
    {
        return $this->role === 'hod';
    }

    public function isTrainer()
    {
        return $this->role === 'trainer';
    }

    public function isExamsOffice()
    {
        return $this->role === 'exams_office';
    }

    public function isStudent()
    {
        return $this->role === 'student';
    }

    public function isValidator()
    {
        return $this->role === 'validator';
    }

    // Status check methods
    public function isActive()
    {
        return $this->status === 'active';
    }

    public function isInactive()
    {
        return $this->status === 'inactive';
    }

    public function isSuspended()
    {
        return $this->status === 'suspended';
    }

    // Scope methods
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeByRole($query, $role)
    {
        return $query->where('role', $role);
    }

    public function scopeInDepartment($query, $departmentId)
    {
        return $query->where('department_id', $departmentId);
    }
}
