<?php

namespace App\Policies;

use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class UserPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        // Admin can view all users
        // HOD can view users in their department
        // Trainer can view students in their classes
        return $user->isAdmin() || $user->isHOD() || $user->isTrainer() 
            || $user->isExamsOffice() || $user->isValidator();
    }

    /**
     * Determine whether the user can view the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, User $model)
    {
        // Users can always view their own profile
        if ($user->id === $model->id) {
            return true;
        }
        
        // Admin can view any
        if ($user->isAdmin()) {
            return true;
        }
        
        // HOD can view users in their department
        if ($user->isHOD() && $model->department_id === $user->department_id) {
            return true;
        }
        
        // Trainer can view students in their classes
        if ($user->isTrainer()) {
            // Check if model is a student enrolled in any of trainer's classes
            // This would require additional logic based on class assignments
            return $model->isStudent();
        }
        
        return false;
    }

    /**
     * Determine whether the user can create models.
     *
     * @param  \App\Models\User  $user
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        // Only admin can create users
        return $user->isAdmin();
    }

    /**
     * Determine whether the user can update the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, User $model)
    {
        // Users can update their own profile (limited fields)
        if ($user->id === $model->id) {
            return true;
        }
        
        // Admin can update any user
        return $user->isAdmin();
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, User $model)
    {
        // Only admin can delete users
        // Cannot delete yourself
        return $user->isAdmin() && $user->id !== $model->id;
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, User $model)
    {
        return $user->isAdmin();
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @param  \App\Models\User  $user
     * @param  \App\Models\User  $model
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, User $model)
    {
        return $user->isAdmin() && $user->id !== $model->id;
    }
}
