<?php

namespace App\Services;

use Illuminate\Support\Facades\Redis;
use Illuminate\Support\Facades\Cache;

/**
 * Redis Cache Service
 * 
 * Provides structured caching with consistent key naming and expiration policies
 * for E-PoE system performance optimization.
 */
class RedisCacheService
{
    /**
     * Key prefix for all cache keys
     */
    const KEY_PREFIX = 'epoe:';

    /**
     * Cache expiration times (in seconds)
     */
    const TTL_SHORT = 180;      // 3 minutes - for frequently changing data
    const TTL_MEDIUM = 600;     // 10 minutes - for dashboard stats
    const TTL_LONG = 3600;      // 1 hour - for relatively static data
    const TTL_VERY_LONG = 86400; // 24 hours - for term-level data

    /**
     * Get HOD dashboard statistics
     * 
     * @param int $hodId
     * @param int $termId
     * @return array
     */
    public static function getHodDashboardStats(int $hodId, int $termId): array
    {
        $key = self::KEY_PREFIX . "hod:dashboard:stats:{$hodId}:term:{$termId}";
        
        return Cache::remember($key, self::TTL_MEDIUM, function() use ($hodId, $termId) {
            // This will be populated by the controller
            return [];
        });
    }

    /**
     * Set HOD dashboard statistics
     * 
     * @param int $hodId
     * @param int $termId
     * @param array $stats
     * @return void
     */
    public static function setHodDashboardStats(int $hodId, int $termId, array $stats): void
    {
        $key = self::KEY_PREFIX . "hod:dashboard:stats:{$hodId}:term:{$termId}";
        Cache::put($key, $stats, self::TTL_MEDIUM);
    }

    /**
     * Get unit marks summary
     * 
     * @param int $unitId
     * @param int $termId
     * @return array|null
     */
    public static function getUnitMarksSummary(int $unitId, int $termId): ?array
    {
        $key = self::KEY_PREFIX . "unit:marks:summary:{$unitId}:term:{$termId}";
        return Cache::get($key);
    }

    /**
     * Set unit marks summary
     * 
     * @param int $unitId
     * @param int $termId
     * @param array $summary
     * @return void
     */
    public static function setUnitMarksSummary(int $unitId, int $termId, array $summary): void
    {
        $key = self::KEY_PREFIX . "unit:marks:summary:{$unitId}:term:{$termId}";
        Cache::put($key, $summary, self::TTL_MEDIUM);
    }

    /**
     * Get class performance summary
     * 
     * @param int $classId
     * @param int $termId
     * @return array|null
     */
    public static function getClassPerformanceSummary(int $classId, int $termId): ?array
    {
        $key = self::KEY_PREFIX . "class:performance:{$classId}:term:{$termId}";
        return Cache::get($key);
    }

    /**
     * Set class performance summary
     * 
     * @param int $classId
     * @param int $termId
     * @param array $summary
     * @return void
     */
    public static function setClassPerformanceSummary(int $classId, int $termId, array $summary): void
    {
        $key = self::KEY_PREFIX . "class:performance:{$classId}:term:{$termId}";
        Cache::put($key, $summary, self::TTL_MEDIUM);
    }

    /**
     * Get term-level statistics
     * 
     * @param int $termId
     * @return array|null
     */
    public static function getTermStatistics(int $termId): ?array
    {
        $key = self::KEY_PREFIX . "term:stats:{$termId}";
        return Cache::get($key);
    }

    /**
     * Set term-level statistics
     * 
     * @param int $termId
     * @param array $stats
     * @return void
     */
    public static function setTermStatistics(int $termId, array $stats): void
    {
        $key = self::KEY_PREFIX . "term:stats:{$termId}";
        Cache::put($key, $stats, self::TTL_VERY_LONG);
    }

    /**
     * Get department classes for HOD
     * 
     * @param int $hodId
     * @param int $termId
     * @return array|null
     */
    public static function getHodDepartmentClasses(int $hodId, int $termId): ?array
    {
        $key = self::KEY_PREFIX . "hod:department:classes:{$hodId}:term:{$termId}";
        return Cache::get($key);
    }

    /**
     * Set department classes for HOD
     * 
     * @param int $hodId
     * @param int $termId
     * @param array $classIds
     * @return void
     */
    public static function setHodDepartmentClasses(int $hodId, int $termId, array $classIds): void
    {
        $key = self::KEY_PREFIX . "hod:department:classes:{$hodId}:term:{$termId}";
        Cache::put($key, $classIds, self::TTL_LONG);
    }

    /**
     * Invalidate cache for a specific unit
     * 
     * @param int $unitId
     * @param int|null $termId
     * @return void
     */
    public static function invalidateUnitCache(int $unitId, ?int $termId = null): void
    {
        if ($termId) {
            $key = self::KEY_PREFIX . "unit:marks:summary:{$unitId}:term:{$termId}";
            Cache::forget($key);
        } else {
            // Invalidate all term variations
            $pattern = self::KEY_PREFIX . "unit:marks:summary:{$unitId}:term:*";
            self::invalidateByPattern($pattern);
        }
    }

    /**
     * Invalidate cache for a specific class
     * 
     * @param int $classId
     * @param int|null $termId
     * @return void
     */
    public static function invalidateClassCache(int $classId, ?int $termId = null): void
    {
        if ($termId) {
            $key = self::KEY_PREFIX . "class:performance:{$classId}:term:{$termId}";
            Cache::forget($key);
        } else {
            $pattern = self::KEY_PREFIX . "class:performance:{$classId}:term:*";
            self::invalidateByPattern($pattern);
        }
    }

    /**
     * Invalidate cache for a specific term
     * 
     * @param int $termId
     * @return void
     */
    public static function invalidateTermCache(int $termId): void
    {
        $patterns = [
            self::KEY_PREFIX . "term:stats:{$termId}",
            self::KEY_PREFIX . "hod:*:term:{$termId}",
            self::KEY_PREFIX . "unit:*:term:{$termId}",
            self::KEY_PREFIX . "class:*:term:{$termId}",
        ];

        foreach ($patterns as $pattern) {
            self::invalidateByPattern($pattern);
        }
    }

    /**
     * Invalidate cache by pattern (for Redis)
     * 
     * @param string $pattern
     * @return void
     */
    protected static function invalidateByPattern(string $pattern): void
    {
        try {
            $keys = Redis::keys($pattern);
            if (!empty($keys)) {
                Redis::del($keys);
            }
        } catch (\Exception $e) {
            // Fallback to file cache if Redis is not available
            \Log::warning('Redis pattern invalidation failed: ' . $e->getMessage());
        }
    }

    /**
     * Clear all E-PoE cache
     * 
     * @return void
     */
    public static function clearAll(): void
    {
        try {
            $keys = Redis::keys(self::KEY_PREFIX . '*');
            if (!empty($keys)) {
                Redis::del($keys);
            }
        } catch (\Exception $e) {
            \Log::warning('Redis clear all failed: ' . $e->getMessage());
        }
    }

    /**
     * Get cache statistics
     * 
     * @return array
     */
    public static function getCacheStats(): array
    {
        try {
            $keys = Redis::keys(self::KEY_PREFIX . '*');
            return [
                'total_keys' => count($keys),
                'memory_usage' => Redis::info('memory')['used_memory_human'] ?? 'N/A',
            ];
        } catch (\Exception $e) {
            return [
                'total_keys' => 0,
                'memory_usage' => 'N/A',
                'error' => $e->getMessage(),
            ];
        }
    }
}

