<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * @return void
     */
    public function up()
    {
        // Composite indexes for poe_submissions
        Schema::table('poe_submissions', function (Blueprint $table) {
            // Status + Class (most common filter combination)
            if (!$this->indexExists('poe_submissions', 'idx_submissions_status_class')) {
                DB::statement('CREATE INDEX idx_submissions_status_class ON poe_submissions(status, class_id)');
            }
            
            // Student + Status (for student dashboards)
            if (!$this->indexExists('poe_submissions', 'idx_submissions_student_status')) {
                DB::statement('CREATE INDEX idx_submissions_student_status ON poe_submissions(student_id, status)');
            }
            
            // Class + Status + Submitted At (for listing with sorting)
            if (!$this->indexExists('poe_submissions', 'idx_submissions_class_status_submitted')) {
                DB::statement('CREATE INDEX idx_submissions_class_status_submitted ON poe_submissions(class_id, status, submitted_at DESC)');
            }
            
            // Unit + Class (for unit-based queries)
            if (!$this->indexExists('poe_submissions', 'idx_submissions_unit_class')) {
                DB::statement('CREATE INDEX idx_submissions_unit_class ON poe_submissions(unit_id, class_id)');
            }
        });

        // Composite indexes for classes
        Schema::table('classes', function (Blueprint $table) {
            // Term + Department (most common filter)
            if (!$this->indexExists('classes', 'idx_classes_term_department')) {
                DB::statement('CREATE INDEX idx_classes_term_department ON classes(term_id, department_id)');
            }
            
            // Term + Level
            if (!$this->indexExists('classes', 'idx_classes_term_level')) {
                DB::statement('CREATE INDEX idx_classes_term_level ON classes(term_id, level_id)');
            }
            
            // Term + Department + Level (complete filter)
            if (!$this->indexExists('classes', 'idx_classes_term_dept_level')) {
                DB::statement('CREATE INDEX idx_classes_term_dept_level ON classes(term_id, department_id, level_id)');
            }
        });

        // Composite indexes for enrollments
        Schema::table('enrollments', function (Blueprint $table) {
            // Class + Status
            if (!$this->indexExists('enrollments', 'idx_enrollments_class_status')) {
                DB::statement('CREATE INDEX idx_enrollments_class_status ON enrollments(class_id, status)');
            }
            
            // Student + Status
            if (!$this->indexExists('enrollments', 'idx_enrollments_student_status')) {
                DB::statement('CREATE INDEX idx_enrollments_student_status ON enrollments(student_id, status)');
            }
        });

        // Indexes for poe_evidence
        Schema::table('poe_evidence', function (Blueprint $table) {
            // Submission + File Type (for filtering evidence by type)
            if (!$this->indexExists('poe_evidence', 'idx_evidence_submission_type')) {
                DB::statement('CREATE INDEX idx_evidence_submission_type ON poe_evidence(poe_submission_id, file_type)');
            }
        });

        // Indexes for full-text search (PostgreSQL)
        if (DB::getDriverName() === 'pgsql') {
            // Full-text search on users
            if (!$this->indexExists('users', 'idx_users_search')) {
                DB::statement("CREATE INDEX idx_users_search ON users USING gin(to_tsvector('english', coalesce(name, '') || ' ' || coalesce(email, '')))");
            }
            
            // Full-text search on units
            if (!$this->indexExists('units', 'idx_units_search')) {
                DB::statement("CREATE INDEX idx_units_search ON units USING gin(to_tsvector('english', coalesce(name, '') || ' ' || coalesce(code, '')))");
            }
        }
    }

    /**
     * Reverse the migrations.
     *
     * @return void
     */
    public function down()
    {
        Schema::table('poe_submissions', function (Blueprint $table) {
            DB::statement('DROP INDEX IF EXISTS idx_submissions_status_class');
            DB::statement('DROP INDEX IF EXISTS idx_submissions_student_status');
            DB::statement('DROP INDEX IF EXISTS idx_submissions_class_status_submitted');
            DB::statement('DROP INDEX IF EXISTS idx_submissions_unit_class');
        });

        Schema::table('classes', function (Blueprint $table) {
            DB::statement('DROP INDEX IF EXISTS idx_classes_term_department');
            DB::statement('DROP INDEX IF EXISTS idx_classes_term_level');
            DB::statement('DROP INDEX IF EXISTS idx_classes_term_dept_level');
        });

        Schema::table('enrollments', function (Blueprint $table) {
            DB::statement('DROP INDEX IF EXISTS idx_enrollments_class_status');
            DB::statement('DROP INDEX IF EXISTS idx_enrollments_student_status');
        });

        Schema::table('poe_evidence', function (Blueprint $table) {
            DB::statement('DROP INDEX IF EXISTS idx_evidence_submission_type');
        });

        if (DB::getDriverName() === 'pgsql') {
            DB::statement('DROP INDEX IF EXISTS idx_users_search');
            DB::statement('DROP INDEX IF EXISTS idx_units_search');
        }
    }

    /**
     * Check if index exists
     */
    protected function indexExists($table, $index)
    {
        $driver = DB::getDriverName();
        
        if ($driver === 'pgsql') {
            $result = DB::selectOne(
                "SELECT COUNT(*) as count FROM pg_indexes WHERE tablename = ? AND indexname = ?",
                [$table, $index]
            );
            return $result->count > 0;
        }
        
        return false;
    }
};

