@extends('layouts.student')

@section('title', 'Edit Submission')
@section('page-title', 'Edit Submission')

@section('content')
    <div class="mb-4 sm:mb-6">
        <a href="{{ route('student.submissions.show', $submission) }}" class="text-indigo-600 hover:text-indigo-900 flex items-center touch-manipulation">
            <svg class="h-5 w-5 mr-1 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
            </svg>
            Back to Submission
        </a>
    </div>

    <div class="bg-white shadow-sm rounded-lg overflow-hidden">
        <div class="px-4 sm:px-6 py-4 border-b border-gray-200 bg-gray-50">
            <h3 class="text-base sm:text-lg font-semibold text-gray-900">Edit Submission</h3>
            <p class="mt-1 text-sm text-gray-600">Update your submission details and evidence files</p>
        </div>

        <form method="POST" action="{{ route('student.submissions.update', $submission) }}" enctype="multipart/form-data" class="px-4 sm:px-6 py-4 sm:py-6">
            @csrf
            @method('PUT')

            <!-- Submission Info (Read-only) -->
            <div class="mb-6 p-4 bg-gray-50 rounded-lg">
                <h4 class="text-sm font-medium text-gray-700 mb-2">Submission Information</h4>
                <dl class="grid grid-cols-2 gap-4 text-sm">
                    <div>
                        <dt class="text-gray-500">Unit</dt>
                        <dd class="font-medium text-gray-900">{{ $submission->unit->name }} @if($submission->unit->code)({{ $submission->unit->code }})@endif</dd>
                    </div>
                    <div>
                        <dt class="text-gray-500">Class</dt>
                        <dd class="font-medium text-gray-900">{{ $submission->schoolClass->name }} @if($submission->schoolClass->code)({{ $submission->schoolClass->code }})@endif</dd>
                    </div>
                </dl>
            </div>

            <!-- Notes -->
            <div class="mb-6">
                <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">
                    Notes (Optional)
                </label>
                <textarea name="notes" id="notes" rows="4" class="w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 @error('notes') border-red-300 @enderror" placeholder="Add any additional notes or comments about your submission...">{{ old('notes', $submission->notes) }}</textarea>
                @error('notes')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
                <p class="mt-1 text-xs text-gray-500">Maximum 1000 characters</p>
            </div>

            <!-- Existing Evidence Files -->
            <div class="mb-6">
                <label class="block text-sm font-medium text-gray-700 mb-2">
                    Existing Evidence Files
                </label>
                @if($submission->evidence->count() > 0)
                    <div class="space-y-2">
                        @foreach($submission->evidence as $evidence)
                            <div class="flex items-center justify-between p-3 border border-gray-200 rounded-lg">
                                <div class="flex items-center space-x-3">
                                    <input type="checkbox" name="remove_evidence[]" value="{{ $evidence->id }}" id="remove_{{ $evidence->id }}" class="rounded border-gray-300 text-red-600 focus:ring-red-500">
                                    <label for="remove_{{ $evidence->id }}" class="flex-1 cursor-pointer">
                                        <p class="text-sm font-medium text-gray-900">{{ $evidence->file_name }}</p>
                                        <p class="text-xs text-gray-500">{{ ucfirst($evidence->file_type) }} • {{ number_format($evidence->file_size / 1024, 2) }} KB</p>
                                    </label>
                                </div>
                                <a href="{{ route('student.submissions.evidence.download', $evidence) }}" class="text-indigo-600 hover:text-indigo-900 text-sm">
                                    Download
                                </a>
                            </div>
                        @endforeach
                    </div>
                    <p class="mt-2 text-xs text-gray-500">Check the boxes above to remove files</p>
                @else
                    <p class="text-sm text-gray-500">No evidence files uploaded yet</p>
                @endif
            </div>

            <!-- Add New Files -->
            <div class="mb-6" id="media-evidence-edit-wrap">
                <label for="files" class="block text-sm font-medium text-gray-700 mb-2">
                    Add New Evidence Files (Optional)
                </label>
                <input type="file" name="files[]" id="files" multiple accept=".pdf,.doc,.docx,.jpg,.jpeg,.png,.mp4,.mp3,.wav" class="block w-full text-sm text-gray-500 file:mr-4 file:py-2 file:px-4 file:rounded-md file:border-0 file:text-sm file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100 @error('files') border-red-300 @enderror" data-max-bytes="10485760">
                @error('files')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
                @error('files.*')
                    <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                @enderror
                <p class="mt-1 text-xs text-gray-500">
                    Allowed file types: PDF, DOC, DOCX, JPG, JPEG, PNG, MP4, MP3, WAV. Maximum file size: 10MB per file. You can select multiple files.
                </p>
                <p id="files-chosen-edit" class="mt-2 text-sm text-indigo-600 hidden" aria-live="polite"></p>
            </div>

            <!-- Submit Button -->
            <div class="flex flex-col-reverse sm:flex-row items-stretch sm:items-center justify-end gap-3 pt-4 border-t border-gray-200">
                <a href="{{ route('student.submissions.show', $submission) }}" class="inline-flex items-center justify-center px-4 py-2.5 min-h-[44px] border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 touch-manipulation">
                    Cancel
                </a>
                <button type="submit" class="inline-flex items-center justify-center px-4 py-2.5 min-h-[44px] border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 touch-manipulation">
                    Update Submission
                </button>
            </div>
        </form>
    </div>

    <div class="mt-6 bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div class="flex">
            <svg class="w-5 h-5 text-yellow-600 mr-2 mt-0.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>
            </svg>
            <div>
                <h4 class="text-sm font-semibold text-yellow-900">Note</h4>
                <p class="mt-1 text-sm text-yellow-700">
                    You can only edit draft submissions. After submitting, you will not be able to make changes.
                </p>
            </div>
        </div>
    </div>

    @push('scripts')
    <script>
    (function() {
        var MAX_SIZE_BYTES = 10 * 1024 * 1024; // 10MB
        var MAX_SIZE_LABEL = '10MB';
        var fileInput = document.getElementById('files');
        var form = fileInput && fileInput.closest('form');
        var filesChosenEl = document.getElementById('files-chosen-edit');

        function formatBytes(bytes) {
            if (bytes < 1024) return bytes + ' B';
            if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
            return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
        }

        function getOversizedFiles(files) {
            var list = [];
            for (var i = 0; i < files.length; i++) {
                if (files[i].size > MAX_SIZE_BYTES) {
                    list.push({ name: files[i].name, size: files[i].size });
                }
            }
            return list;
        }

        function showSizeWarning(oversized) {
            var msg = 'The following file(s) exceed the maximum size of ' + MAX_SIZE_LABEL + ' per file:\n\n';
            oversized.forEach(function(f) {
                msg += '• ' + f.name + ' (' + formatBytes(f.size) + ')\n';
            });
            msg += '\nPlease choose smaller files or remove them from the selection.';
            alert(msg);
        }

        function updateFilesChosenLabel() {
            if (!filesChosenEl || !fileInput) return;
            var files = fileInput.files;
            if (files.length === 0) {
                filesChosenEl.classList.add('hidden');
                filesChosenEl.textContent = '';
                return;
            }
            var oversized = getOversizedFiles(files);
            if (oversized.length > 0) {
                showSizeWarning(oversized);
                fileInput.value = '';
                filesChosenEl.classList.add('hidden');
                filesChosenEl.textContent = '';
                return;
            }
            filesChosenEl.classList.remove('hidden');
            filesChosenEl.textContent = files.length === 1
                ? '1 file chosen: ' + files[0].name
                : files.length + ' files chosen: ' + Array.from(files).map(function(f) { return f.name; }).join(', ');
        }

        if (fileInput) {
            fileInput.addEventListener('change', function() {
                updateFilesChosenLabel();
            });
        }

        if (form) {
            form.addEventListener('submit', function(e) {
                var files = fileInput.files;
                if (files.length === 0) return;
                var oversized = getOversizedFiles(files);
                if (oversized.length > 0) {
                    e.preventDefault();
                    showSizeWarning(oversized);
                    return false;
                }
            });
        }
    })();
    </script>
    @endpush
@endsection

