<?php

use App\Http\Controllers\ProfileController;
use App\Services\DatabaseHealthCheck;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

// Database health check endpoint (no middleware to avoid circular dependency)
Route::get('/health/database', function () {
    $health = DatabaseHealthCheck::check();
    return response()->json($health, $health['status'] === 'connected' ? 200 : 503);
})->name('health.database');

// Homepage is now the login page (handled in auth.php for guest middleware)

// Dashboard redirects based on role
Route::get('/dashboard', function () {
    $user = auth()->user();
    
    if ($user->isAdmin()) {
        return redirect()->route('admin.dashboard');
    } elseif ($user->isHOD()) {
        return redirect()->route('hod.dashboard');
    } elseif ($user->isTrainer()) {
        return redirect()->route('trainer.dashboard');
    } elseif ($user->isExamsOffice()) {
        return redirect()->route('exams.dashboard');
    } elseif ($user->isValidator()) {
        return redirect()->route('validator.dashboard');
    } elseif ($user->isStudent()) {
        return redirect()->route('student.dashboard');
    }
    
    return view('dashboard');
})->middleware(['auth', 'verified', 'active'])->name('dashboard');

// Role-based route groups
Route::middleware(['auth', 'verified', 'active', 'role.admin'])->prefix('admin')->name('admin.')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Admin\DashboardController::class, 'index'])->name('dashboard');
    
    // User Management
    Route::get('/users', [App\Http\Controllers\Admin\UserController::class, 'index'])->name('users.index');
    Route::get('/users/create', [App\Http\Controllers\Admin\UserController::class, 'create'])->name('users.create');
    Route::post('/users', [App\Http\Controllers\Admin\UserController::class, 'store'])->name('users.store');
    Route::get('/users/{user}', [App\Http\Controllers\Admin\UserController::class, 'show'])->name('users.show');
    Route::get('/users/{user}/edit', [App\Http\Controllers\Admin\UserController::class, 'edit'])->name('users.edit');
    Route::put('/users/{user}', [App\Http\Controllers\Admin\UserController::class, 'update'])->name('users.update');
    Route::delete('/users/{user}', [App\Http\Controllers\Admin\UserController::class, 'destroy'])->name('users.destroy');
    
    // Academic Management
    Route::get('/terms', [App\Http\Controllers\Admin\TermController::class, 'index'])->name('terms.index');
    Route::get('/terms/create', [App\Http\Controllers\Admin\TermController::class, 'create'])->name('terms.create');
    Route::post('/terms', [App\Http\Controllers\Admin\TermController::class, 'store'])->name('terms.store');
    Route::get('/terms/{term}', [App\Http\Controllers\Admin\TermController::class, 'show'])->name('terms.show');
    Route::get('/terms/{term}/edit', [App\Http\Controllers\Admin\TermController::class, 'edit'])->name('terms.edit');
    Route::put('/terms/{term}', [App\Http\Controllers\Admin\TermController::class, 'update'])->name('terms.update');
    Route::delete('/terms/{term}', [App\Http\Controllers\Admin\TermController::class, 'destroy'])->name('terms.destroy');
    Route::post('/terms/{term}/activate', [App\Http\Controllers\Admin\TermController::class, 'activate'])->name('terms.activate');
    Route::post('/terms/{term}/close', [App\Http\Controllers\Admin\TermController::class, 'close'])->name('terms.close');
    // Department Management
    Route::get('/departments', [App\Http\Controllers\Admin\DepartmentController::class, 'index'])->name('departments.index');
    Route::get('/departments/create', [App\Http\Controllers\Admin\DepartmentController::class, 'create'])->name('departments.create');
    Route::post('/departments', [App\Http\Controllers\Admin\DepartmentController::class, 'store'])->name('departments.store');
    Route::get('/departments/{department}', [App\Http\Controllers\Admin\DepartmentController::class, 'show'])->name('departments.show');
    Route::get('/departments/{department}/edit', [App\Http\Controllers\Admin\DepartmentController::class, 'edit'])->name('departments.edit');
    Route::put('/departments/{department}', [App\Http\Controllers\Admin\DepartmentController::class, 'update'])->name('departments.update');
    Route::delete('/departments/{department}', [App\Http\Controllers\Admin\DepartmentController::class, 'destroy'])->name('departments.destroy');
    
    // Level Management
    Route::get('/levels', [App\Http\Controllers\Admin\LevelController::class, 'index'])->name('levels.index');
    Route::get('/levels/create', [App\Http\Controllers\Admin\LevelController::class, 'create'])->name('levels.create');
    Route::post('/levels', [App\Http\Controllers\Admin\LevelController::class, 'store'])->name('levels.store');
    Route::get('/levels/{level}', [App\Http\Controllers\Admin\LevelController::class, 'show'])->name('levels.show');
    Route::get('/levels/{level}/edit', [App\Http\Controllers\Admin\LevelController::class, 'edit'])->name('levels.edit');
    Route::put('/levels/{level}', [App\Http\Controllers\Admin\LevelController::class, 'update'])->name('levels.update');
    Route::delete('/levels/{level}', [App\Http\Controllers\Admin\LevelController::class, 'destroy'])->name('levels.destroy');
    
    // Class Management
    Route::get('/classes', [App\Http\Controllers\Admin\ClassController::class, 'index'])->name('classes.index');
    Route::get('/classes/create', [App\Http\Controllers\Admin\ClassController::class, 'create'])->name('classes.create');
    Route::post('/classes', [App\Http\Controllers\Admin\ClassController::class, 'store'])->name('classes.store');
    Route::get('/classes/{class}', [App\Http\Controllers\Admin\ClassController::class, 'show'])->name('classes.show');
    Route::get('/classes/{class}/edit', [App\Http\Controllers\Admin\ClassController::class, 'edit'])->name('classes.edit');
    Route::put('/classes/{class}', [App\Http\Controllers\Admin\ClassController::class, 'update'])->name('classes.update');
    Route::delete('/classes/{class}', [App\Http\Controllers\Admin\ClassController::class, 'destroy'])->name('classes.destroy');
    Route::post('/classes/{class}/assign-unit', [App\Http\Controllers\Admin\ClassController::class, 'assignUnit'])->name('classes.assign-unit');
    Route::post('/classes/{class}/remove-unit', [App\Http\Controllers\Admin\ClassController::class, 'removeUnit'])->name('classes.remove-unit');
    
    // Unit Management
    Route::get('/units', [App\Http\Controllers\Admin\UnitController::class, 'index'])->name('units.index');
    Route::get('/units/create', [App\Http\Controllers\Admin\UnitController::class, 'create'])->name('units.create');
    Route::post('/units', [App\Http\Controllers\Admin\UnitController::class, 'store'])->name('units.store');
    Route::get('/units/{unit}', [App\Http\Controllers\Admin\UnitController::class, 'show'])->name('units.show');
    Route::get('/units/{unit}/edit', [App\Http\Controllers\Admin\UnitController::class, 'edit'])->name('units.edit');
    Route::put('/units/{unit}', [App\Http\Controllers\Admin\UnitController::class, 'update'])->name('units.update');
    Route::delete('/units/{unit}', [App\Http\Controllers\Admin\UnitController::class, 'destroy'])->name('units.destroy');
    Route::post('/units/{unit}/assign-class', [App\Http\Controllers\Admin\UnitController::class, 'assignToClass'])->name('units.assign-class');
    Route::post('/units/{unit}/remove-class', [App\Http\Controllers\Admin\UnitController::class, 'removeFromClass'])->name('units.remove-class');
    Route::post('/units/{unit}/assign-trainer', [App\Http\Controllers\Admin\UnitController::class, 'assignTrainer'])->name('units.assign-trainer');
    Route::post('/units/{unit}/remove-trainer', [App\Http\Controllers\Admin\UnitController::class, 'removeTrainer'])->name('units.remove-trainer');
    
    // Unit Allocations (Trainer-Unit-Class assignments)
    Route::get('/unit-allocations', [App\Http\Controllers\Admin\UnitAllocationController::class, 'index'])->name('unit-allocations.index');
    Route::get('/unit-allocations/create', [App\Http\Controllers\Admin\UnitAllocationController::class, 'create'])->name('unit-allocations.create');
    Route::post('/unit-allocations', [App\Http\Controllers\Admin\UnitAllocationController::class, 'store'])->name('unit-allocations.store');
    Route::delete('/unit-allocations/{id}', [App\Http\Controllers\Admin\UnitAllocationController::class, 'destroy'])->name('unit-allocations.destroy');
    
    // PoE Management
    Route::get('/submissions', [App\Http\Controllers\Admin\SubmissionController::class, 'index'])->name('submissions.index');
    Route::get('/submissions/{submission}', [App\Http\Controllers\Admin\SubmissionController::class, 'show'])->name('submissions.show');
    Route::put('/submissions/{submission}/status', [App\Http\Controllers\Admin\SubmissionController::class, 'updateStatus'])->name('submissions.update-status');
    Route::delete('/submissions/{submission}', [App\Http\Controllers\Admin\SubmissionController::class, 'destroy'])->name('submissions.destroy');
    Route::get('/submissions/evidence/{evidence}/download', [App\Http\Controllers\Admin\SubmissionController::class, 'downloadEvidence'])->name('submissions.evidence.download');
    Route::delete('/submissions/evidence/{evidence}', [App\Http\Controllers\Admin\SubmissionController::class, 'deleteEvidence'])->name('submissions.evidence.destroy');
    
    // Reports
    Route::get('/reports', [App\Http\Controllers\Admin\ReportController::class, 'index'])->name('reports.index');
    Route::get('/reports/submissions', [App\Http\Controllers\Admin\ReportController::class, 'submissions'])->name('reports.submissions');
    Route::get('/reports/student/{student}', [App\Http\Controllers\Admin\ReportController::class, 'student'])->name('reports.student');
    Route::get('/reports/class/{class}', [App\Http\Controllers\Admin\ReportController::class, 'class'])->name('reports.class');
    Route::get('/reports/department/{department}', [App\Http\Controllers\Admin\ReportController::class, 'department'])->name('reports.department');
    
    // History (Past Terms)
    Route::get('/history', [App\Http\Controllers\Admin\HistoryController::class, 'index'])->name('history.index');
    Route::get('/history/{term}', [App\Http\Controllers\Admin\HistoryController::class, 'show'])->name('history.show');
    Route::get('/history/{term}/classes', [App\Http\Controllers\Admin\HistoryController::class, 'classes'])->name('history.classes');
    Route::get('/history/{term}/units', [App\Http\Controllers\Admin\HistoryController::class, 'units'])->name('history.units');
    Route::get('/history/{term}/submissions', [App\Http\Controllers\Admin\HistoryController::class, 'submissions'])->name('history.submissions');
    Route::get('/history/{term}/trainers', [App\Http\Controllers\Admin\HistoryController::class, 'trainers'])->name('history.trainers');
    Route::get('/history/{term}/practical-evidence', [App\Http\Controllers\Admin\HistoryController::class, 'practicalEvidence'])->name('history.practical-evidence');
    Route::get('/history/{term}/units/{unit}/print-marks', [App\Http\Controllers\Admin\HistoryController::class, 'printUnitMarks'])->name('history.print-unit-marks');
    
    // System
    Route::get('/settings', [App\Http\Controllers\Admin\SettingController::class, 'index'])->name('settings.index');
    Route::post('/settings/institution', [App\Http\Controllers\Admin\SettingController::class, 'updateInstitution'])->name('settings.institution');
    Route::post('/settings/system', [App\Http\Controllers\Admin\SettingController::class, 'updateSystem'])->name('settings.system');
    Route::post('/settings/notifications', [App\Http\Controllers\Admin\SettingController::class, 'updateNotifications'])->name('settings.notifications');
    Route::post('/settings/files', [App\Http\Controllers\Admin\SettingController::class, 'updateFiles'])->name('settings.files');
    Route::delete('/settings/logo', [App\Http\Controllers\Admin\SettingController::class, 'deleteLogo'])->name('settings.logo.delete');
    Route::delete('/settings/favicon', [App\Http\Controllers\Admin\SettingController::class, 'deleteFavicon'])->name('settings.favicon.delete');
    Route::post('/settings/security', [App\Http\Controllers\Admin\SettingController::class, 'updateSecurity'])->name('settings.security');
    Route::post('/settings/backup/create', [App\Http\Controllers\Admin\SettingController::class, 'createBackup'])->name('settings.backup.create');
    Route::post('/settings/backup/restore', [App\Http\Controllers\Admin\SettingController::class, 'restoreBackup'])->name('settings.backup.restore');
    Route::get('/settings/backup/download/{filename}', [App\Http\Controllers\Admin\SettingController::class, 'downloadBackup'])->name('settings.backup.download');
    Route::delete('/settings/backup/{filename}', [App\Http\Controllers\Admin\SettingController::class, 'deleteBackup'])->name('settings.backup.delete');
    Route::post('/settings/clear-data', [App\Http\Controllers\Admin\SettingController::class, 'clearAllData'])->name('settings.clear-data');
    Route::get('/audit-logs', [App\Http\Controllers\Admin\AuditLogController::class, 'index'])->name('audit-logs.index');
    Route::get('/audit-logs/export', [App\Http\Controllers\Admin\AuditLogController::class, 'export'])->name('audit-logs.export');
    Route::get('/audit-logs/{auditLog}', [App\Http\Controllers\Admin\AuditLogController::class, 'show'])->name('audit-logs.show');
    
    // Practical Marks Compilation
    Route::get('/assignments/{assignment}/practical-marks', [App\Http\Controllers\Admin\PracticalMarksController::class, 'index'])->name('practical-marks.index');
    Route::post('/assignments/{assignment}/practical-marks/lock', [App\Http\Controllers\Admin\PracticalMarksController::class, 'lockResults'])->name('practical-marks.lock');
    Route::post('/assignments/{assignment}/practical-marks/unlock', [App\Http\Controllers\Admin\PracticalMarksController::class, 'unlockResults'])->name('practical-marks.unlock');
    Route::post('/assignments/{assignment}/practical-marks/release', [App\Http\Controllers\Admin\PracticalMarksController::class, 'releaseResults'])->name('practical-marks.release');
    Route::get('/assignments/{assignment}/practical-marks/export/{format}', [App\Http\Controllers\Admin\PracticalMarksController::class, 'export'])->name('practical-marks.export');
});

Route::middleware(['auth', 'verified', 'active', 'role.hod'])->prefix('hod')->name('hod.')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\HOD\DashboardController::class, 'index'])->name('dashboard');
    
    // Approval Management
    Route::get('/approvals', [App\Http\Controllers\HOD\ApprovalController::class, 'index'])->name('approvals.index');
    Route::get('/approvals/{submission}', [App\Http\Controllers\HOD\ApprovalController::class, 'show'])->name('approvals.show');
    Route::post('/approvals/{submission}/approve', [App\Http\Controllers\HOD\ApprovalController::class, 'approve'])->name('approvals.approve');
    Route::post('/approvals/{submission}/reject', [App\Http\Controllers\HOD\ApprovalController::class, 'reject'])->name('approvals.reject');
    
    // Portfolio Evidence (Practical Evidence)
    Route::get('/practical-evidence', [App\Http\Controllers\HOD\PracticalEvidenceController::class, 'index'])->name('practical-evidence.index');
    Route::get('/practical-evidence/{practicalEvidence}', [App\Http\Controllers\HOD\PracticalEvidenceController::class, 'show'])->name('practical-evidence.show');
    Route::post('/practical-evidence/{practicalEvidence}/comment', [App\Http\Controllers\HOD\PracticalEvidenceController::class, 'addComment'])->name('practical-evidence.comment');
    Route::get('/practical-evidence/{practicalEvidence}/download', [App\Http\Controllers\HOD\PracticalEvidenceController::class, 'download'])->name('practical-evidence.download');
    
    // Submissions (Legacy - kept for backward compatibility)
    Route::get('/submissions', [App\Http\Controllers\HOD\ApprovalController::class, 'index'])->name('submissions.index');
    Route::get('/submissions/{submission}', [App\Http\Controllers\HOD\ApprovalController::class, 'show'])->name('submissions.show');
    Route::get('/submissions/evidence/{evidence}/download', [App\Http\Controllers\Admin\SubmissionController::class, 'downloadEvidence'])->name('submissions.evidence.download');
    
    // Units
    Route::get('/units', [App\Http\Controllers\HOD\UnitController::class, 'index'])->name('units.index');
    Route::get('/units/departmental', [App\Http\Controllers\HOD\UnitController::class, 'departmental'])->name('units.departmental');
    Route::get('/units/create', [App\Http\Controllers\HOD\UnitController::class, 'create'])->name('units.create');
    Route::post('/units', [App\Http\Controllers\HOD\UnitController::class, 'store'])->name('units.store');
    Route::get('/units/{unit}', [App\Http\Controllers\HOD\UnitController::class, 'show'])->name('units.show');
    Route::get('/units/{unit}/configure', [App\Http\Controllers\HOD\UnitController::class, 'configure'])->name('units.configure');
    Route::post('/units/{unit}/configure', [App\Http\Controllers\HOD\UnitController::class, 'storeConfiguration'])->name('units.configure.store');
    Route::post('/units/{unit}/assignments', [App\Http\Controllers\HOD\AssignmentController::class, 'store'])->name('units.assignments.store');
    Route::get('/assignments/{assignment}', [App\Http\Controllers\HOD\AssignmentController::class, 'show'])->name('assignments.show');
    Route::get('/assignments/{assignment}/edit', [App\Http\Controllers\HOD\AssignmentController::class, 'edit'])->name('assignments.edit');
    Route::put('/assignments/{assignment}', [App\Http\Controllers\HOD\AssignmentController::class, 'update'])->name('assignments.update');
    Route::delete('/assignments/{assignment}', [App\Http\Controllers\HOD\AssignmentController::class, 'destroy'])->name('assignments.destroy');
    Route::get('/assignments/{assignment}/attachment/{index}', [App\Http\Controllers\HOD\AssignmentController::class, 'downloadAttachment'])->name('assignments.attachment.download');
    Route::get('/assignments/{assignment}/tool/{toolType}/{index}', [App\Http\Controllers\HOD\AssignmentController::class, 'downloadTool'])->name('assignments.tool.download');
    Route::get('/assignments/{assignment}/submissions/{submission}', [App\Http\Controllers\HOD\AssignmentController::class, 'showSubmission'])->name('assignments.submissions.show');
    Route::post('/assignments/{assignment}/submissions/{submission}/mark', [App\Http\Controllers\HOD\AssignmentController::class, 'markSubmission'])->name('assignments.submissions.mark');

    // Assessor Tool (Mark Practical - same as Trainer)
    Route::get('/assignments/{assignment}/assessor-tool/structure', [App\Http\Controllers\HOD\AssessorToolController::class, 'createStructure'])->name('assessor-tool.create-structure');
    Route::post('/assignments/{assignment}/assessor-tool/structure', [App\Http\Controllers\HOD\AssessorToolController::class, 'storeStructure'])->name('assessor-tool.store-structure');
    Route::get('/assignments/{assignment}/assessor-tool/marking', [App\Http\Controllers\HOD\AssessorToolController::class, 'marking'])->name('assessor-tool.marking');
    Route::get('/assignments/{assignment}/assessor-tool/marking/{submissionId}', [App\Http\Controllers\HOD\AssessorToolController::class, 'marking'])->name('assessor-tool.marking.submission');
    Route::post('/assignments/{assignment}/submissions/{submission}/assessor-tool/marks', [App\Http\Controllers\HOD\AssessorToolController::class, 'saveMarks'])->name('assessor-tool.save-marks');
    Route::post('/assignments/{assignment}/assessor-tool/marks', [App\Http\Controllers\HOD\AssessorToolController::class, 'saveMarks'])->name('assessor-tool.save-marks-new')->where('submission', '.*');
    Route::get('/assignments/{assignment}/submissions/{submission}/assessor-tool/generate', [App\Http\Controllers\HOD\AssessorToolController::class, 'generateDocument'])->name('assessor-tool.generate');

    // Marks Management
    Route::get('/marks', [App\Http\Controllers\HOD\MarksController::class, 'index'])->name('marks.index');
    Route::get('/marks/units/{unit}/print', [App\Http\Controllers\HOD\MarksController::class, 'printUnitMarks'])->name('marks.print-unit');
    Route::get('/marks/classes/{class}/print', [App\Http\Controllers\HOD\MarksController::class, 'printClassMarks'])->name('marks.print-class');
    Route::post('/marks/classes/{class}/units/{unit}/push-to-validation', [App\Http\Controllers\HOD\MarksController::class, 'pushToValidation'])->name('marks.push-to-validation');
    Route::post('/marks/validation-requests/{validationRequest}/cancel', [App\Http\Controllers\HOD\MarksController::class, 'cancelValidationRequest'])->name('marks.validation-requests.cancel');
    
    // User Management
    Route::resource('users', App\Http\Controllers\HOD\UserController::class);
    
    // Classes Management
    Route::resource('classes', App\Http\Controllers\HOD\ClassController::class);
    Route::post('/classes/{class}/assign-unit', [App\Http\Controllers\HOD\ClassController::class, 'assignUnit'])->name('classes.assign-unit');
    Route::delete('/classes/{class}/remove-unit', [App\Http\Controllers\HOD\ClassController::class, 'removeUnit'])->name('classes.remove-unit');
    
    // Unit Allocations
    Route::get('/unit-allocations', [App\Http\Controllers\HOD\UnitAllocationController::class, 'index'])->name('unit-allocations.index');
    Route::get('/unit-allocations/create', [App\Http\Controllers\HOD\UnitAllocationController::class, 'create'])->name('unit-allocations.create');
    Route::post('/unit-allocations', [App\Http\Controllers\HOD\UnitAllocationController::class, 'store'])->name('unit-allocations.store');
    Route::post('/unit-allocations/allocate-to-class', [App\Http\Controllers\HOD\UnitAllocationController::class, 'allocateToClass'])->name('unit-allocations.allocate-to-class');
    
    // Notifications
    Route::get('/notifications', [App\Http\Controllers\HOD\NotificationController::class, 'index'])->name('notifications.index');
    Route::post('/notifications/individual', [App\Http\Controllers\HOD\NotificationController::class, 'sendIndividual'])->name('notifications.individual');
    Route::post('/notifications/department', [App\Http\Controllers\HOD\NotificationController::class, 'sendToDepartment'])->name('notifications.department');
    Route::post('/notifications/class', [App\Http\Controllers\HOD\NotificationController::class, 'sendToClass'])->name('notifications.class');
    
    // Reports
    Route::get('/reports', [App\Http\Controllers\HOD\ReportController::class, 'index'])->name('reports.index');
    Route::get('/reports/department', [App\Http\Controllers\HOD\ReportController::class, 'department'])->name('reports.department');
    Route::get('/reports/class', [App\Http\Controllers\HOD\ReportController::class, 'class'])->name('reports.class');
    
    // History (Past Terms)
    Route::get('/history', [App\Http\Controllers\HOD\HistoryController::class, 'index'])->name('history.index');
    Route::get('/history/{term}', [App\Http\Controllers\HOD\HistoryController::class, 'show'])->name('history.show');
    Route::get('/history/{term}/classes', [App\Http\Controllers\HOD\HistoryController::class, 'classes'])->name('history.classes');
    Route::get('/history/{term}/units', [App\Http\Controllers\HOD\HistoryController::class, 'units'])->name('history.units');
    Route::get('/history/{term}/submissions', [App\Http\Controllers\HOD\HistoryController::class, 'submissions'])->name('history.submissions');
    Route::get('/history/{term}/trainers', [App\Http\Controllers\HOD\HistoryController::class, 'trainers'])->name('history.trainers');
    Route::get('/history/{term}/practical-evidence', [App\Http\Controllers\HOD\HistoryController::class, 'practicalEvidence'])->name('history.practical-evidence');
    Route::get('/history/{term}/units/{unit}/print-marks', [App\Http\Controllers\HOD\HistoryController::class, 'printUnitMarks'])->name('history.print-unit-marks');
    
    // Practical Marks Compilation
    Route::get('/assignments/{assignment}/practical-marks', [App\Http\Controllers\HOD\PracticalMarksController::class, 'index'])->name('practical-marks.index');
    Route::post('/assignments/{assignment}/practical-marks/lock', [App\Http\Controllers\HOD\PracticalMarksController::class, 'lockResults'])->name('practical-marks.lock');
    Route::post('/assignments/{assignment}/practical-marks/unlock', [App\Http\Controllers\HOD\PracticalMarksController::class, 'unlockResults'])->name('practical-marks.unlock');
    Route::post('/assignments/{assignment}/practical-marks/release', [App\Http\Controllers\HOD\PracticalMarksController::class, 'releaseResults'])->name('practical-marks.release');
    Route::get('/assignments/{assignment}/practical-marks/export/{format}', [App\Http\Controllers\HOD\PracticalMarksController::class, 'export'])->name('practical-marks.export');
    Route::post('/assignments/{assignment}/practical-marks/mark-absent', [App\Http\Controllers\HOD\PracticalMarksController::class, 'markAbsent'])->name('practical-marks.mark-absent');
    Route::post('/assignments/{assignment}/practical-marks/remove-absent', [App\Http\Controllers\HOD\PracticalMarksController::class, 'removeAbsent'])->name('practical-marks.remove-absent');
    
    // Messages
    Route::get('/messages', [App\Http\Controllers\HOD\MessageController::class, 'index'])->name('messages.index');
    Route::get('/messages/{message}', [App\Http\Controllers\HOD\MessageController::class, 'show'])->name('messages.show');
    
    // Profile
    Route::get('/profile', [App\Http\Controllers\HOD\ProfileController::class, 'show'])->name('profile.show');
    Route::get('/profile/edit', [App\Http\Controllers\HOD\ProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [App\Http\Controllers\HOD\ProfileController::class, 'update'])->name('profile.update');
});

Route::middleware(['auth', 'verified', 'active', 'role.trainer'])->prefix('trainer')->name('trainer.')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Trainer\DashboardController::class, 'index'])->name('dashboard');
    
    // Profile
    Route::get('/profile', [App\Http\Controllers\Trainer\ProfileController::class, 'show'])->name('profile.show');
    Route::get('/profile/edit', [App\Http\Controllers\Trainer\ProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [App\Http\Controllers\Trainer\ProfileController::class, 'update'])->name('profile.update');
    
    // Units
    Route::get('/units', [App\Http\Controllers\Trainer\UnitController::class, 'index'])->name('units.index');
    Route::get('/units/{unit}', [App\Http\Controllers\Trainer\UnitController::class, 'show'])->name('units.show');
    
    // Portfolio Evidence (Task-based)
    Route::get('/portfolio-evidence', [App\Http\Controllers\Trainer\PortfolioEvidenceController::class, 'index'])->name('portfolio-evidence.index');
    Route::get('/portfolio-evidence/{task}', [App\Http\Controllers\Trainer\PortfolioEvidenceController::class, 'show'])->name('portfolio-evidence.show');
    Route::put('/portfolio-evidence/{task}/status', [App\Http\Controllers\Trainer\PortfolioEvidenceController::class, 'updateStatus'])->name('portfolio-evidence.update-status');
    Route::get('/portfolio-evidence/evidence/{evidence}/download', [App\Http\Controllers\Trainer\PortfolioEvidenceController::class, 'download'])->name('portfolio-evidence.download');
    
    
    // Assignments
    Route::get('/assignments/{assignment}', [App\Http\Controllers\Trainer\AssignmentController::class, 'show'])->name('assignments.show');
    Route::get('/assignments/{assignment}/edit', [App\Http\Controllers\Trainer\AssignmentController::class, 'edit'])->name('assignments.edit');
    Route::post('/units/{unit}/assignments', [App\Http\Controllers\Trainer\AssignmentController::class, 'store'])->name('assignments.store');
    Route::put('/assignments/{assignment}', [App\Http\Controllers\Trainer\AssignmentController::class, 'update'])->name('assignments.update');
    Route::delete('/assignments/{assignment}', [App\Http\Controllers\Trainer\AssignmentController::class, 'destroy'])->name('assignments.destroy');
    Route::get('/assignments/{assignment}/attachment/{index}', [App\Http\Controllers\Trainer\AssignmentController::class, 'downloadAttachment'])->name('assignments.attachment.download');
    Route::get('/assignments/{assignment}/tool/{toolType}/{index}', [App\Http\Controllers\Trainer\AssignmentController::class, 'downloadTool'])->name('assignments.tool.download');
    Route::get('/assignments/{assignment}/submissions/{submission}', [App\Http\Controllers\Trainer\AssignmentController::class, 'showSubmission'])->name('assignments.submissions.show');
    Route::post('/assignments/{assignment}/submissions/{submission}/mark', [App\Http\Controllers\Trainer\AssignmentController::class, 'markSubmission'])->name('assignments.submissions.mark');
    
    // Assessor Tool Management (for Practical Assignments)
    Route::get('/assignments/{assignment}/assessor-tool/structure', [App\Http\Controllers\Trainer\AssessorToolController::class, 'createStructure'])->name('assessor-tool.create-structure');
    Route::post('/assignments/{assignment}/assessor-tool/structure', [App\Http\Controllers\Trainer\AssessorToolController::class, 'storeStructure'])->name('assessor-tool.store-structure');
    Route::get('/assignments/{assignment}/assessor-tool/marking', [App\Http\Controllers\Trainer\AssessorToolController::class, 'marking'])->name('assessor-tool.marking');
    Route::get('/assignments/{assignment}/assessor-tool/marking/{submissionId}', [App\Http\Controllers\Trainer\AssessorToolController::class, 'marking'])->name('assessor-tool.marking.submission');
    Route::post('/assignments/{assignment}/submissions/{submission}/assessor-tool/marks', [App\Http\Controllers\Trainer\AssessorToolController::class, 'saveMarks'])->name('assessor-tool.save-marks');
    Route::post('/assignments/{assignment}/assessor-tool/marks', [App\Http\Controllers\Trainer\AssessorToolController::class, 'saveMarks'])->name('assessor-tool.save-marks-new')->where('submission', '.*');
    Route::get('/assignments/{assignment}/submissions/{submission}/assessor-tool/generate', [App\Http\Controllers\Trainer\AssessorToolController::class, 'generateDocument'])->name('assessor-tool.generate');
    
    // Practical Marks Compilation
    Route::get('/assignments/{assignment}/practical-marks', [App\Http\Controllers\Trainer\PracticalMarksController::class, 'index'])->name('practical-marks.index');
    Route::post('/assignments/{assignment}/practical-marks/mark-absent', [App\Http\Controllers\Trainer\PracticalMarksController::class, 'markAbsent'])->name('practical-marks.mark-absent');
    Route::post('/assignments/{assignment}/practical-marks/remove-absent', [App\Http\Controllers\Trainer\PracticalMarksController::class, 'removeAbsent'])->name('practical-marks.remove-absent');
    Route::get('/assignments/{assignment}/practical-marks/export/{format}', [App\Http\Controllers\Trainer\PracticalMarksController::class, 'export'])->name('practical-marks.export');
    
    
    // Submissions
    Route::get('/submissions', [App\Http\Controllers\Trainer\SubmissionController::class, 'index'])->name('submissions.index');
    Route::get('/submissions/{submission}', [App\Http\Controllers\Trainer\SubmissionController::class, 'show'])->name('submissions.show');
    
    // Marks Management
    Route::get('/marks', [App\Http\Controllers\Trainer\MarksController::class, 'index'])->name('marks.index');
    Route::get('/marks/units/{unit}', [App\Http\Controllers\Trainer\MarksController::class, 'showUnit'])->name('marks.show-unit');
    Route::put('/marks/{submission}', [App\Http\Controllers\Trainer\MarksController::class, 'update'])->name('marks.update');
    Route::get('/marks/assignments/{assignment}/print', [App\Http\Controllers\Trainer\MarksController::class, 'printAssignment'])->name('marks.print-assignment');
    Route::get('/marks/units/{unit}/print', [App\Http\Controllers\Trainer\MarksController::class, 'printUnitMarks'])->name('marks.print-unit');
    Route::post('/marks/classes/{class}/units/{unit}/push-to-validation', [App\Http\Controllers\Trainer\MarksController::class, 'pushClassToValidation'])->name('marks.push-to-validation');
    Route::post('/marks/validation-requests/{validationRequest}/cancel', [App\Http\Controllers\Trainer\MarksController::class, 'cancelValidationRequest'])->name('marks.validation-requests.cancel');

    // Messages
    Route::get('/messages', [App\Http\Controllers\Trainer\MessageController::class, 'index'])->name('messages.index');
    Route::post('/messages/notifications/{notification}/read', [App\Http\Controllers\Trainer\MessageController::class, 'markNotificationAsRead'])->name('messages.notification.read');
    Route::post('/messages/{message}/read', [App\Http\Controllers\Trainer\MessageController::class, 'markMessageAsRead'])->name('messages.read');
    Route::get('/messages/notifications/{notification}', [App\Http\Controllers\Trainer\MessageController::class, 'showNotification'])->name('messages.show-notification');
    Route::get('/messages/{message}', [App\Http\Controllers\Trainer\MessageController::class, 'showMessage'])->name('messages.show');
    
    // Notifications (Send to Students)
    Route::get('/notifications', [App\Http\Controllers\Trainer\NotificationController::class, 'index'])->name('notifications.index');
    Route::post('/notifications/individual', [App\Http\Controllers\Trainer\NotificationController::class, 'sendIndividual'])->name('notifications.individual');
    Route::post('/notifications/class', [App\Http\Controllers\Trainer\NotificationController::class, 'sendToClass'])->name('notifications.class');
    Route::post('/notifications/all', [App\Http\Controllers\Trainer\NotificationController::class, 'sendToAll'])->name('notifications.all');
});

Route::middleware(['auth', 'verified', 'active', 'role.exams_office'])->prefix('exams')->name('exams.')->group(function () {
    Route::get('/dashboard', function () {
        return view('exams.dashboard');
    })->name('dashboard');
    // Exams Office routes will be added here
});

Route::middleware(['auth', 'verified', 'active', 'role.validator'])->prefix('validator')->name('validator.')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Validator\DashboardController::class, 'index'])->name('dashboard');
    
    // Submissions (Validations)
    Route::get('/submissions', [App\Http\Controllers\Validator\SubmissionController::class, 'index'])->name('submissions.index');
    Route::get('/submissions/{submission}', [App\Http\Controllers\Validator\SubmissionController::class, 'show'])->name('submissions.show');
    Route::post('/submissions/{submission}/validate', [App\Http\Controllers\Validator\SubmissionController::class, 'submitValidation'])->name('submissions.validate');
    
    // Class Validation Requests
    Route::get('/validation-requests', [App\Http\Controllers\Validator\ValidationRequestController::class, 'index'])->name('validation-requests.index');
    Route::get('/validation-requests/classes/{class}/units/{unit}', [App\Http\Controllers\Validator\ValidationRequestController::class, 'show'])->name('validation-requests.show');
    Route::get('/validation-requests/classes/{class}/units/{unit}/print-marks', [App\Http\Controllers\Validator\ValidationRequestController::class, 'showPrintMarks'])->name('validation-requests.print-marks');
    Route::post('/validation-requests/{validationRequest}/validate', [App\Http\Controllers\Validator\ValidationRequestController::class, 'submitValidation'])->name('validation-requests.validate');

    // Portfolio evidence review per validation request
    Route::get('/validation-requests/{validationRequest}/portfolio', [App\Http\Controllers\Validator\ValidationRequestController::class, 'portfolioIndex'])->name('validation-requests.portfolio');
    Route::get('/validation-requests/{validationRequest}/portfolio/students/{student}', [App\Http\Controllers\Validator\ValidationRequestController::class, 'portfolioStudent'])->name('validation-requests.portfolio.student');
    Route::get('/validation-requests/{validationRequest}/portfolio/students/{student}/exams', [App\Http\Controllers\Validator\ValidationRequestController::class, 'portfolioStudentExams'])->name('validation-requests.portfolio.student.exams');
    Route::get('/validation-requests/{validationRequest}/portfolio/students/{student}/practicals', [App\Http\Controllers\Validator\ValidationRequestController::class, 'portfolioStudentPracticals'])->name('validation-requests.portfolio.student.practicals');
    
    // Evidence viewing & download for validation requests (used by validator.units.show JS)
    Route::get('/units/{validationRequest}/evidence/{student}', [App\Http\Controllers\Validator\UnitController::class, 'getStudentEvidence'])->name('units.evidence');
    Route::get('/units/evidence/{file}/download', [App\Http\Controllers\Validator\UnitController::class, 'downloadEvidence'])->name('units.evidence.download');
    
    // Practical Evidence
    Route::get('/units/{unit}/students/{student}/practical-evidence', [App\Http\Controllers\Validator\PracticalEvidenceController::class, 'show'])->name('practical-evidence.show');
    Route::post('/practical-evidence/{practicalEvidence}/comment', [App\Http\Controllers\Validator\PracticalEvidenceController::class, 'addComment'])->name('practical-evidence.comment');
    Route::get('/practical-evidence/{practicalEvidence}/download', [App\Http\Controllers\Validator\PracticalEvidenceController::class, 'download'])->name('practical-evidence.download');
});

Route::middleware(['auth', 'verified', 'active', 'role.student'])->prefix('student')->name('student.')->group(function () {
    Route::get('/dashboard', [App\Http\Controllers\Student\DashboardController::class, 'index'])->name('dashboard');
    
    // Profile
    Route::get('/profile', [App\Http\Controllers\Student\ProfileController::class, 'edit'])->name('profile.edit');
    Route::put('/profile', [App\Http\Controllers\Student\ProfileController::class, 'update'])->name('profile.update');
    
    // Student Submissions
    Route::get('/submissions', [App\Http\Controllers\Student\SubmissionController::class, 'index'])->name('submissions.index');
    Route::get('/submissions/create', [App\Http\Controllers\Student\SubmissionController::class, 'create'])->name('submissions.create');
    Route::post('/submissions', [App\Http\Controllers\Student\SubmissionController::class, 'store'])->name('submissions.store');
    Route::get('/submissions/{submission}', [App\Http\Controllers\Student\SubmissionController::class, 'show'])->name('submissions.show');
    Route::get('/submissions/{submission}/edit', [App\Http\Controllers\Student\SubmissionController::class, 'edit'])->name('submissions.edit');
    Route::put('/submissions/{submission}', [App\Http\Controllers\Student\SubmissionController::class, 'update'])->name('submissions.update');
    Route::delete('/submissions/{submission}', [App\Http\Controllers\Student\SubmissionController::class, 'destroy'])->name('submissions.destroy');
    Route::post('/submissions/{submission}/submit', [App\Http\Controllers\Student\SubmissionController::class, 'submit'])->name('submissions.submit');
    Route::get('/submissions/evidence/{evidence}/download', [App\Http\Controllers\Student\SubmissionController::class, 'downloadEvidence'])->name('submissions.evidence.download');
    
    // Student Units
    Route::get('/units', [App\Http\Controllers\Student\UnitController::class, 'index'])->name('units.index');
    Route::get('/units/{unit}', [App\Http\Controllers\Student\UnitController::class, 'show'])->name('units.show');
    
    // Student Assignments
    Route::get('/assignments/{assignment}', [App\Http\Controllers\Student\AssignmentController::class, 'show'])->name('assignments.show');
    Route::post('/assignments/{assignment}/submit', [App\Http\Controllers\Student\AssignmentController::class, 'submit'])->name('assignments.submit');
    Route::get('/assignments/{assignment}/attachment/{index}', [App\Http\Controllers\Student\AssignmentController::class, 'downloadAttachment'])->name('assignments.attachment.download');
    Route::get('/assignments/{assignment}/tool/{toolType}/{index}', [App\Http\Controllers\Student\AssignmentController::class, 'downloadTool'])->name('assignments.tool.download');
    
    // Student Practical Marks (view own marks only)
    Route::get('/assignments/{assignment}/practical-marks', [App\Http\Controllers\Student\AssignmentController::class, 'viewPracticalMarks'])->name('assignments.practical-marks');
    
    // Student Progress
    Route::get('/progress', [App\Http\Controllers\Student\ProgressController::class, 'index'])->name('progress.index');
    Route::get('/progress/unit-document-pdf', [App\Http\Controllers\Student\ProgressController::class, 'unitDocumentPdf'])->name('progress.unit-document-pdf');
    Route::get('/progress/units-checklist-pdf', [App\Http\Controllers\Student\ProgressController::class, 'unitsChecklistPdf'])->name('progress.units-checklist-pdf');
    Route::get('/progress/units-exams-pdf', [App\Http\Controllers\Student\ProgressController::class, 'unitsExamsPdf'])->name('progress.units-exams-pdf');
    
    // Student Messages & Notifications
    Route::get('/messages', [App\Http\Controllers\Student\MessageController::class, 'index'])->name('messages.index');
    Route::get('/messages/notification/{notification}', [App\Http\Controllers\Student\MessageController::class, 'showNotification'])->name('messages.show-notification');
    Route::get('/messages/message/{message}', [App\Http\Controllers\Student\MessageController::class, 'showMessage'])->name('messages.show-message');
    Route::post('/messages/notification/{notification}/read', [App\Http\Controllers\Student\MessageController::class, 'markNotificationAsRead'])->name('messages.mark-notification-read');
    Route::post('/messages/message/{message}/read', [App\Http\Controllers\Student\MessageController::class, 'markMessageAsRead'])->name('messages.mark-message-read');
    Route::post('/messages/notifications/mark-all-read', [App\Http\Controllers\Student\MessageController::class, 'markAllNotificationsAsRead'])->name('messages.mark-all-notifications-read');
    Route::post('/messages/messages/mark-all-read', [App\Http\Controllers\Student\MessageController::class, 'markAllMessagesAsRead'])->name('messages.mark-all-messages-read');
    
});

// Common authenticated routes
Route::middleware(['auth', 'active'])->group(function () {
    Route::get('/profile', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/profile', [ProfileController::class, 'update'])->name('profile.update');
    Route::delete('/profile', [ProfileController::class, 'destroy'])->name('profile.destroy');
});

require __DIR__.'/auth.php';
